-- FROM F14:
----- sidewinders
-- from aim9_family.lua
dofile("Scripts/Database/Weapons/warheads.lua")
dofile("Scripts/Database/generated_wstypes.lua")

local POUNDS_TO_KG = 0.453592
local INCH_TO_METER = 0.0254

-- Below is the data pulled from the famous CBU sheet.

local suu7_tube_number = 19
local suu7_dispenser_empty_mass = 121 * POUNDS_TO_KG

-- https://www.bulletpicker.com/pdf/CBU.pdf#page=6
-- BLU-4 (CBU-1A/A)
-- N = 508
-- Explosive Mass Total = 89.8 lbs
-- Explosive Mass Per Bomblet = 0.177 lbs
-- Mass = 1.18 lbs
-- Length = 4.92 ing
-- Diameter = 2.76 in
local blu4 = {
	N_per_tube 		= 27, -- 27x19 = 513
	N_effect 		= 10,
	mass 			= 1.18 * POUNDS_TO_KG,
	explosive_mass 	= 0.177 * POUNDS_TO_KG,
	length 			= 4.92 * INCH_TO_METER,
	diameter 		= 2.76 * INCH_TO_METER,
	model_name = "HB_F-4E_EXT_BLU4B_armed",
	name = "BLU-4B",
}

-- BLU-3 (CBU-2/A)
-- N = 360
-- Explosive Mass Total = 129.07 lbs
-- Explosive Mass Per Bomblet = 0.359 lbs
-- Mass = 1.73 lbs
-- Length = 3.75 in
-- Diameter = 2.75 in
local blu3 = {
	N_per_tube 		= 19, -- 19x19 = 361
	N_effect 		= 10,
	mass 			= 1.73 * POUNDS_TO_KG,
	explosive_mass 	= 0.359 * POUNDS_TO_KG,
	length 			= 3.75 * INCH_TO_METER,
	diameter 		= 2.75 * INCH_TO_METER,
	model_name = "HB_F-4E_EXT_BLU3B_armed",
	name = "BLU-3",
}

-- BLU-3B (CBU-2B/A)
-- N = 406
-- Explosive Mass Total = 143.06 lbs
-- Explosive Mass Per Bomblet = 0.359 lbs
-- Mass = 1.73 lbs
-- Length = 3.75 in
-- Diameter = 2.75 in
local blu3b = {
	N_per_tube 		= 22, -- 22x19 = 418
	N_effect 		= 10,
	mass 			= 1.73 * POUNDS_TO_KG,
	explosive_mass 	= 0.359 * POUNDS_TO_KG,
	length 			= 3.75 * INCH_TO_METER,
	diameter 		= 2.75 * INCH_TO_METER,
	model_name = "HB_F-4E_EXT_BLU3B_armed",
	name = "BLU-3B",
}

function create_blux_weapon(data)
	return {
		category = CAT_CLUSTER_DESC,
		scheme = {
			cluster = {
				mass        = data.mass,    -- Empty weight with warhead, W/O fuel, kg
				caliber     = 3.0 * data.diameter,  -- 3x blu diameter (account for brakes)
				cx_coeff    = {1.000000, 0.290000, 0.710000, 0.140000, 1.280000},  -- Cx
				L           = data.length,   -- Length, meters (95mm / 3.75")
				I           = 0.09, --0.083 * data.mass * data.length * data.length,   -- kgm2 - moment of inertia - I = 1/12 ML2
				Ma          = 0.03,  -- dependence moment coefficient of  by  AoA angular acceleration  T / I   (??? can't solve for torque w/o knowing rotation)
				Mw          = 0.3, --  rad/s  - 57.3°/s - dependence moment coefficient by angular velocity (|V|*sin(?))/|r| -  Mw  =  Ma * t   (???)
				wind_sigma  = 20, -- dispersion coefficient  mk81=80, mk82=80, mk83=150, mk84=220 ... heavier = harder to push with wind?
				count = data.N_per_tube,
				effect_count = data.N_effect,
				impulse_sigma = 2,
				model_name = data.model_name,
				moment_sigma = 0.0001,
			},
			warhead         = {
				mass        = data.mass,
				expl_mass   = data.explosive_mass,
				other_factors = {1.0, 1.0, 1.0};
				obj_factors = {1, 1};
				concrete_factors = {1.0, 1.0, 1.0};
				cumulative_factor = 1;
				concrete_obj_factor = 1.0;
				cumulative_thickness = 1.0;
				piercing_mass = data.explosive_mass * 0.5;
				caliber     = data.diameter;
			},
		},
		display_name = "Bomblets BLU-3B",
		mass = 0, -- can't remember why
		model = "",
		name = data.name,
		cluster_scheme = "cluster",
		type_name = "cluster",
		wsTypeOfWeapon = {wsType_Weapon, wsType_Bomb, wsType_Bomb_Cluster, WSTYPE_PLACEHOLDER},
	}
end

function make_cluster_group(bomblets, model)
    local name = bomblets.name.."_GROUP"
    local model_name = model-- bomblets.scheme.cluster.model_name
    local display_name = bomblets.display_name.." x "..bomblets.scheme.cluster.count..", HE"
    local mass = bomblets.scheme.cluster.mass
    local fm = bomblets.scheme.cluster
    local descriptor = bomblets.descriptor
    local group = {
        category        = CAT_BOMBS,
        wsTypeOfWeapon    = {wsType_Weapon, wsType_Bomb, wsType_Bomb_Cluster, WSTYPE_PLACEHOLDER},
        VyHold            = -100.0,
        Ag                = -1.23,
        type            = 5,
        mass            = mass,
        hMin            = 30,
        hMax            = 1000.0,
        Cx                = 0.0,--0.000413, -- Do not enable drag on these, otherwise u get negative drag, because the dispenser does not change drag.
        name            = name,
        model            = model_name,
        user_name        = display_name,
        displayName = _(display_name),
        scheme            = "bomb-cassette",
        class_name        = "wAmmunition",
        fm = fm,
        control = {
            open_delay = 0,
        },
        launcher = {
            blocks    = {"cluster"},
            cluster    = descriptor,
        },
        targeting_data = {
            v0 = 200,
            char_time = 26,
            bomblet_char_time = 26,
        },
        puff = {
            show_puff = 0
        },
        shape_table_data = {{
            name        = model_name,
            file        = model_name,
            username    = display_name,
            index        = WSTYPE_PLACEHOLDER,
        }}
    }
    return group
end

BLU_3 = create_blux_weapon(blu3)
BLU_3B = create_blux_weapon(blu3b)
BLU_4B = create_blux_weapon(blu4)
declare_weapon(BLU_3)
declare_weapon(BLU_3B)
declare_weapon(BLU_4B)


BLU_3_NEW_GROUP = make_cluster_group(BLU_3, "HB_F-4E_EXT_BLU3B_unarmed")
BLU_3B_NEW_GROUP = make_cluster_group(BLU_3B, "HB_F-4E_EXT_BLU3B_unarmed")
BLU_4B_NEW_GROUP = make_cluster_group(BLU_4B, "HB_F-4E_EXT_BLU4B_unarmed")


declare_weapon(BLU_3_NEW_GROUP)
declare_weapon(BLU_3B_NEW_GROUP)
declare_weapon(BLU_4B_NEW_GROUP)

local bomblet_data =
{
    ["BLU-3"] = { 
		name = BLU_3_NEW_GROUP.displayName, 
		mass = BLU_3_NEW_GROUP.mass * blu3.N_per_tube, 
		wstype = BLU_3_NEW_GROUP.wsTypeOfWeapon, 
		model = "HB_F-4E_EXT_BLU3B_unarmed"
	},
    ["BLU-3B"] = { 
		name = 
		BLU_3B_NEW_GROUP.displayName, 
		mass = BLU_3B_NEW_GROUP.mass * blu3b.N_per_tube, 
		wstype = BLU_3B_NEW_GROUP.wsTypeOfWeapon, 
		model = "HB_F-4E_EXT_BLU3B_unarmed"
	},
    ["BLU-4"] = { 
		name = BLU_4B_NEW_GROUP.displayName,
		mass = BLU_4B_NEW_GROUP.mass * blu4.N_per_tube,
		wstype = BLU_4B_NEW_GROUP.wsTypeOfWeapon,
		model = "HB_F-4E_EXT_BLU4B_unarmed"
	},
}

function make_suu7_dispenser(dispenser,element,count,display_count,display_name) -- assemble a cluster dispenser
    
    display_name = display_name or dispenser
    display_count = display_count or count
    
    local bomblet = bomblet_data[element] or bomblet_data["BLU-3"]
    local data = {}
    data.category           = CAT_BOMBS
    data.CLSID              = "{"..dispenser.."}"
    data.attribute          = {wsType_Weapon, wsType_Bomb, wsType_Container, WSTYPE_PLACEHOLDER}
    data.Picture            = "suu-1a.png"
    data.Count              = count
    data.displayName        = display_name.." pod - "..tostring(display_count).." x tubes of "..bomblet.name
    data.wsTypeOfWeapon     = bomblet.wstype
    data.Weight             = suu7_dispenser_empty_mass + (count * bomblet.mass)
	data.Weight_Empty       = suu7_dispenser_empty_mass
	data.Cx_pil             = 0.001887
    data.Elements           =
    {
        {
            Position    =    {0,    0,    0},
            ShapeName    =    "HB_ORD_SUU_7",
            IsAdapter   =   true,
        },
    }

	shape_table_data = {{
            name        = bomblet.model,
            file        = bomblet.model,
            username    = bomblet.model,
            index        = WSTYPE_PLACEHOLDER,
    }}


	local connector_name = "tube_"


	-- Outer In
	-- Opposing Sides
	local order = {
		2,18,
		3,17,
		7,13,
		12,8,
		16,4,
		19,1,
		6,14,
		11,9,
		15,5,
		10
	}
    for i = 1,19 do
		table.insert(data.Elements, {
            DrawArgs    =    {{0,1}},
            --Position    =    pos_bomblets[(i%19)+1],
            ShapeName    =    bomblet.model,
            --Rotation    =   {0,0,0},
			connector_name = connector_name..tostring(order[20-i]),
        })
    end
    return data
end


CBU_1A 	= make_suu7_dispenser("HB_F4E_CBU-1/A", 	"BLU-4", 	suu7_tube_number)
CBU_2A 	= make_suu7_dispenser("HB_F4E_CBU-2/A", 	"BLU-3", 	suu7_tube_number)
CBU_2BA = make_suu7_dispenser("HB_F4E_CBU-2B/A", 	"BLU-3B", 	suu7_tube_number)
declare_loadout(CBU_1A)
declare_loadout(CBU_2A)
declare_loadout(CBU_2BA)

AN_M81_warhead = simple_warhead(113,207)
AN_M81_improved = HE_penetrating_warhead(113,207)

AGM_12C = {
	category		= CAT_MISSILES,
	wsTypeOfWeapon	= {wsType_Weapon, wsType_Missile, wsType_AS_Missile, WSTYPE_PLACEHOLDER},
	name			= "AGM_12C",
	user_name		= _("AGM-12C"),
	scheme			= "AGM-12",
	class_name		= "wAmmunitionVikhr",
	model			= "HB_F-4E_EXT_AGM12C",

	Escort			= 1,
	Head_Type		= 7,
	sigma			= {4, 4, 4},
	M				= 263,
	H_max			= 10000,
	H_min			= -1,
	Diam			= 305,
	Cx_pil			= 4,		--will need adjustment to match 1.1 drag index on F-4E
	D_max			= 13000,
	D_min			= 1500,
	Head_Form		= 1,
	Life_Time		= 30,
	Nr_max			= 1.2,
	v_min			= 170,
	v_mid			= 300,
	Mach_max		= 1.8,
	t_b				= 0,
	t_acc			= 2.5,
	t_marsh			= 0,
	Range_max		= 13000,
	H_min_t			= 0,
	Fi_start		= 0.1,
	Fi_rak			= 3.14152,
	Fi_excort		= 0.4,
	Fi_search		= 99.9,
	OmViz_max		= 99.9,
	X_back			= -0.922,
	Y_back			= 0,
	Z_back			= 0,
	Reflection		= 0.03,
	KillDistance	= 0,

	booster = {
		impulse							= 230,	-- unspecified DB propelant
		fuel_mass						= 41.88, -- 37.8kN for 2.5s
		work_time						= 2.5,
		nozzle_position					= {{-1, 0, 0.}},
		nozzle_orientationXYZ			= {{0, 0, 0}},
		tail_width						= 0.18,
		smoke_color						= {0.8, 0.8, 0.8},
		smoke_transparency				= 0.01,
		custom_smoke_dissipation_factor	= 1,
		effect_type						= 0
	},

	spiral_nav = {
		t_cone_near_rad			= 10000,
		def_cone_max_dist		= 22000,
		def_cone_near_rad		= 10000,
		def_cone_near_rad_st	= 0,
		def_cone_time_stab_rad	= 0,
		gb_angle				= 0.0036,
		gb_min_dist				= 3200,
		gb_use_time				= 5.3,
		gb_max_retW				= 0.4,
		gb_ret_Kp				= 1.1,
	},

	autopilot = {
		Kp					= 0.07,
		Ki					= 0.01,
		Kd					= 0.07,
		max_ctrl_angle		= 6.24,
		delay				= 2,
		op_time				= 30,
		fins_discreet		= 0.08,
		no_ctrl_center_ang	= 0.00002,
		fins_limit			= 1,

		self_destruct_maneuver			= 1,
		self_destruct_maneuver_cmd_y	= 0.3,
		self_destruct_maneuver_cmd_z	= 0.6,
		self_destruct_start_timer		= 18,
		self_destruct_command_timer		= 3,
		self_destruct_no_signal_lim		= 0.5,
	},

	fm = {
		mass		= 263,
		caliber		= 0.3048,
		L			= 3.4798,
		I			= 1 / 12 * 263 * 3.4798 ^ 2,
		Ma			= 2,
		Mw			= 5,
		cx_coeff	= {1,0.34,0.38,0.536,1.31},
		wind_sigma	= 0.0,
		wind_time	= 0.0,
		Sw			= 0.465,
		dCydA		= {0.07, 0.036},
		A			= 0.36,
		maxAoa		= 0.23,
		finsTau		= 0.01,
		Ma_x		= 3,
		Mw_x		= 1.5,
		I_x			= 10,
		Mx0Ma		= {-0.25, -0.11},
		ideal_fins	= true
	},

	wcs = {
		input_delay = 0.1,
		activation_delay = 0.75,
		lifetime = 30,
		reference_offset = -math.rad(50),
		lead_pulse_sensitivity = 0.5,
		sustain_pulse_sensitity = 1,
		sustain_pulse_sentitivity_exponent = 0.2,
		memory_pulse_sensitivity = 1,
		lead_pulse_deadzone = 0.001,
		sustain_pulse_gain_per_cycle = 0.1,
		command_deflection = 0.2,
		base_pulse_length = 0.05
	},

	fuze = {
		default_arm_delays = { 2 },
		default_function_altitude = 0,
	},

	LaunchDistData =
	{
		19,		10,
	
				100,	150,	200,	250,	300,	350,	400,	450,	500,	550,		
		100,	0,	0,	7000,	8000,	0,	0,	10000,	10500,	11000,	12000,	
		200,	0,	0,	7000,	0,	0,	12000,	0,	0,	0,	0,	
		300,	5000,	6000,	7000,	8500,	10500,	11500,	12500,	11000,	12000,	12500,	
		400,	5500,	6500,	7500,	8500,	10500,	12500,	12500,	11000,	12000,	12500,	
		500,	5500,	6500,	7500,	9000,	10500,	12500,	14500,	15500,	15000,	12500,	
		600,	5500,	6500,	7500,	9000,	11000,	13000,	13000,	15000,	16500,	17000,	
		700,	6000,	6500,	7500,	9500,	11000,	13000,	15500,	15500,	17500,	18500,	
		800,	6000,	6500,	7500,	9500,	11500,	13500,	15000,	15500,	18000,	18500,	
		900,	6000,	6500,	8500,	9500,	11500,	13000,	15000,	16000,	18000,	18500,	
		1000,	6000,	6500,	8000,	10000,	11500,	13500,	15000,	17000,	17500,	19000,	
		2000,	9500,	10500,	11500,	12500,	13500,	15000,	16000,	17500,	18500,	19500,	
		3000,	9500,	10500,	12000,	13000,	14000,	15500,	16500,	18000,	19000,	20000,	
		4000,	10000,	11000,	12000,	13500,	14500,	16000,	17000,	18000,	19500,	20500,	
		5000,	10000,	11000,	12500,	13500,	15000,	16000,	17500,	18500,	19500,	20500,	
		6000,	10000,	11500,	12500,	14000,	15000,	16500,	17500,	18500,	19500,	21000,	
		7000,	10000,	11500,	12500,	14000,	15500,	16500,	17500,	19000,	20000,	21000,	
		8000,	10000,	11500,	12500,	14000,	15500,	16500,	17500,	19000,	20000,	21000,	
		9000,	10000,	11500,	12500,	14000,	15500,	16500,	17500,	19000,	20000,	21000,	
		10000,	9500,	11000,	12500,	14000,	15500,	16500,	17500,	18000,	20000,	21000,	
	
	},

	MinLaunchDistData =
	{
		19,		10,
	
				100,	150,	200,	250,	300,	350,	400,	450,	500,	550,		
		100,	0,	0,	500,	500,	500,	750,	500,	0,	0,	0,	
		200,	2050,	2300,	500,	500,	550,	750,	500,	500,	500,	500,	
		300,	2000,	500,	500,	500,	500,	750,	500,	500,	500,	500,	
		400,	500,	500,	500,	500,	500,	750,	500,	500,	500,	500,	
		500,	500,	500,	500,	500,	500,	750,	500,	500,	500,	500,	
		600,	500,	500,	500,	500,	500,	650,	500,	500,	500,	500,	
		700,	500,	500,	500,	500,	500,	750,	500,	500,	500,	500,	
		800,	500,	500,	500,	500,	550,	800,	500,	500,	500,	500,	
		900,	500,	500,	500,	500,	600,	850,	500,	500,	500,	500,	
		1000,	500,	500,	500,	500,	600,	800,	500,	500,	500,	500,	
		2000,	500,	600,	600,	600,	500,	700,	500,	500,	500,	500,	
		3000,	700,	550,	800,	950,	1200,	500,	750,	650,	500,	500,	
		4000,	800,	700,	700,	850,	850,	900,	800,	700,	650,	550,	
		5000,	1000,	800,	900,	1000,	1400,	1100,	900,	800,	700,	700,	
		6000,	1000,	1000,	1500,	1000,	1500,	1500,	1000,	1000,	1000,	1000,	
		7000,	2000,	1500,	2000,	2500,	1500,	2000,	1500,	1500,	1000,	1500,	
		8000,	2500,	2500,	1500,	3500,	1500,	1500,	2500,	1500,	1500,	1500,	
		9000,	3000,	3000,	3000,	2500,	1500,	2000,	2000,	2500,	1500,	1000,	
		10000,	3500,	3000,	3500,	4500,	3500,	1500,	2500,	3000,	2500,	2000,	
	},		

	shape_table_data = {{
		name		= "HB_F-4E_EXT_AGM12C",
		file		= "HB_F-4E_EXT_AGM12C",
		username	= _("AGM-12C"),
		index		= WSTYPE_PLACEHOLDER
	}},

	warhead		= AN_M81_warhead,
}

declare_weapon(AGM_12C)

declare_loadout({
	category 		= CAT_MISSILES,
	CLSID	 		= "{HB_F4E_AGM_12C}",
	wsTypeOfWeapon	= AGM_12C.wsTypeOfWeapon,
	attribute 		= {4, 7, 32, WSTYPE_PLACEHOLDER},
	Count 			= 1,
	Cx_pil			= AGM_12C.Cx_pil/4096,
	Picture			= "GBU31.png",
	displayName		= _("AGM-12C Bullpup - MCLOS missile"),
	Weight			= AGM_12C.mass,

	JettisonSubmunitionOnly = true,

	shape_table_data = {{
				name = "HB_F-4E_LAU-34",
				file = "HB_F-4E_LAU-34",
				username = "HB_F-4E_LAU-34",
				life = 1,
				fire = {0,1},
				index = WSTYPE_PLACEHOLDER,
	}},

	Elements = {
		{
			Position	=	{0,0,0},
			ShapeName	=	"HB_F-4E_LAU-34",
			IsAdapter   =   true,
		},
		{
			Position	=	{0,	0, 0},
			ShapeName = AGM_12C.model,
			connector_name = "AGM45_ATTACH",
		}
	},

})


local AIM_7_FM =
{
	mass				= 231,  
	caliber				= 0.2,  
	wind_sigma			= 0.0,
	wind_time			= 0.0,
	tail_first			= 1,
	fins_part_val		= 0,
	rotated_fins_inp	= 0,
	delta_max			= math.rad(20),
	draw_fins_conv		= {math.rad(90),1,1},
	L					= 0.2,
	S					= 0.0324,
	Ix					= 3.5,
	Iy					= 127.4,
	Iz					= 127.4,
	
	Mxd					= 0.3 * 57.3,
	Mxw					= -44.5,

	table_scale	= 0.2,
	table_degree_values = 1,
--	Mach	  | 0.0		0.2		0.4		0.6		0.8		1.0		1.2		1.4		1.6		1.8		2.0		2.2		2.4		2.6		2.8		3.0		3.2		3.4		3.6		3.8		4.0	 |
	Cx0 	= {	0.34,	0.34,	0.34,	0.34,	0.35,	1.10,	1.27,	1.23,	1.19,	1.12,	1.05,	1.0,	0.95,	0.91,	0.87,	0.84,	0.81,	0.78,	0.76,	0.74,	0.72 },
	CxB 	= {	0.11,	0.11,	0.11,	0.11,	0.11,	0.18,	0.19,	0.17,	0.16,	0.14,	0.13,	0.12,	0.12,	0.11,	0.11,	0.10,	0.09,	0.09,	0.08,	0.08,	0.07 },
	K1		= { 0.0056,	0.0056,	0.0056,	0.0056,	0.0056,	0.0056,	0.0056,	0.0056,	0.0052,	0.0048,	0.0045,	0.0041,	0.0037,	0.0036,	0.0034,	0.0032,	0.0031,	0.0030,	0.0029,	0.0027,	0.0026 },
	K2		= { 0.0055,	0.0055,	0.0055,	0.0055,	0.0055,	0.0055,	0.0055,	0.0055,	0.0051,	0.0047,	0.0043,	0.0037,	0.0031,	0.0032,	0.0033,	0.0035,	0.0036,	0.0037,	0.0038,	0.0039,	0.0040 },
	Cya		= { 1.14,	1.14,	1.14,	1.14,	1.14,	1.42,	1.46,	1.39,	1.32,	1.15,	1.06,	1.00,	0.94,	0.89,	0.83,	0.78,	0.73,	0.69,	0.65,	0.61,	0.57 },
	Cza		= { 1.14,	1.14,	1.14,	1.14,	1.14,	1.42,	1.46,	1.39,	1.32,	1.15,	1.06,	1.00,	0.94,	0.89,	0.83,	0.78,	0.73,	0.69,	0.65,	0.61,	0.57 },
	Mya		= { -0.5,	-0.5},
	Mza		= { -0.5,	-0.5},
	Myw		= { -2.0,	-2.0},
	Mzw		= { -2.0,	-2.0},
	A1trim	= { 10.0,	10.0},
	A2trim	= { 10.0,	10.0},
		
	model_roll = math.rad(45),
	fins_stall = 1,
}


local AIM_7E =
{
	category		= CAT_AIR_TO_AIR,
	name			= "HB-AIM-7E",
	displayName		= _("AIM-7E Sparrow Semi-Active Radar"),
	user_name		= _("AIM-7E"),
	scheme			= "aa_missile_semi_active",
	class_name		= "wAmmunitionSelfHoming",
	model			= "aim-7",
	wsTypeOfWeapon 	= {wsType_Weapon,wsType_Missile,wsType_AA_Missile,WSTYPE_PLACEHOLDER},
	
	mass			= 230,
	Escort			= 1,
	Head_Type		= 6,
	sigma			= {5.6, 5, 5.6},
	H_max			= 24400.0,
	H_min			= 1.0,
	Diam			= 203.0,
	Cx_pil			= 2.21,
	D_max			= 20000.0,
	D_min			= 1500.0,
	Head_Form		= 1,
	Life_Time		= 90.0,
	Nr_max			= 25,
	v_min			= 140.0,
	v_mid			= 500.0,
	Mach_max		= 3.2,
	t_b				= 0.6,
	t_acc			= 3.3,
	t_marsh			= 11.0,
	Range_max		= 50000.0,
	H_min_t			= 15.0,
	Fi_start		= 0.4,
	Fi_rak			= 3.14152,
	Fi_excort		= 0.8,
	Fi_search		= 0.1,
	OmViz_max		= 0.35,
	exhaust			= {0.78, 0.78, 0.78, 0.3};
	X_back			= -2.0,
	Y_back			= -0.0,
	Z_back			= 0.0, -- -0.1,
	Reflection		= 0.08,
	KillDistance	= 12.0,
	M				= 230,
	
	ccm_k0 = 2.0,
	rad_correction = 0,
	height_error_k = 110, -- если проекция скорости цели на линию визирования меньше этого значения, появляется ошибка
	height_error_max_vel = 150, -- пропорциональный коэффициент
	height_error_max_h = 450, -- максимальная высота, где появляется ошибка
	hoj = 1,

	warhead		= predefined_warhead("AIM_7E"),
	warhead_air = predefined_warhead("AIM_7E"),
	
	shape_table_data =
	{
		{
			name  = "hb_aim_7_E";
			file  = "HB_F-4E_AIM7E";
			life  = 1;
			fire  = { 0, 1};
			username = "HB-AIM-7E";
			index 	 = WSTYPE_PLACEHOLDER,
		},
	},
		
	ModelData = {   58 ,  -- model params count
					0.9 ,   -- characteristic square (характеристическая площадь)
			
					-- параметры зависимости Сx
					0.0125 , -- планка Сx0 на дозвуке ( M << 1)
					0.052 , -- высота пика волнового кризиса
					0.010 , -- крутизна фронта на подходе к волновому кризису
					0.002 , -- планка Cx0 на сверхзвуке ( M >> 1)
					0.5  , -- крутизна спада за волновым кризисом 
					1.2  , -- коэффициент отвала поляры
					
					-- параметры зависимости Cy
					2.20, -- планка Cya на дозвуке ( M << 1)
					1.05, -- планка Cya на сверхзвуке ( M >> 1)
					1.20, -- крутизна спада(фронта) за волновым кризисом  

					0.18, -- ~10 degrees Alfa_max  максимальный балансировачный угол, радианы
					0.00, --угловая скорость создаваймая моментом газовых рулей
						
					--	t_statr   t_b      t_accel  t_march   t_inertial   t_break  t_end
					-1.0,        -1.0,       2.8,     0.0,      0.0,		 0.0,    1.0e9,           -- time interval
					 0.0,         0.0, 	   14.28,     0.0,      0.0,         0.0,    0.0,           -- fuel flow rate in second kg/sec(секундный расход массы топлива кг/сек)
					 0.0,         0.0,   35000.0,     0.0,      0.0,         0.0,    0.0,           -- thrust
					
					 1.0e9, -- таймер самоликвидации, сек
					 75.0, -- время работы энергосистемы
					 0.0, -- абсалютеая высота самоликвидации, м
					 1.5, -- время задержки включения управленя, сек
					 1.0e9, -- дальность до цели в момент пуска, выше которой выполняется маневр набора высоты 
					 1.0e9, -- дальность до цели на трассе, менее которой начинается завершение маневра набора высоты (длжен быть больше чем предылущий параметр) 
					 0.0,  -- синус угла возвышения траектории набора горки
					 50.0, -- продольное ускорения взведения взрывателя
					 0.0, -- модуль скорости сообщаймый катапультным устройством, вышибным зарядом и тд
					 1.19, -- характристика системы САУ-РАКЕТА,  коэф фильтра второго порядка K0
					 1.0, -- характристика системы САУ-РАКЕТА,  коэф фильтра второго порядка K1
					 2.0, -- характристика системы САУ-РАКЕТА,  полоса пропускания контура управления
					 -- ЗРП. Данные для рассчета дальностей пуска (индикация на прицеле)
					 14.0, -- производная дальности по скорости носителя на высоте 1км, ППС
					 -19.0, -- производная дальности по скорости цели на высоте 1км, ЗПС
					 -2.2, -- производная по высоте производной дальности по скорости цели, ЗПС
					 19000, -- дальность ракурс 180(навстречу) град, Н=5000м, V=900км/ч, м
					 4800.0, -- дальность ракурс 0(в догон) град, Н=5000м, V=900км/ч, м
					 26000.0, -- дальность ракурс 180(навстречу) град, Н=10000м, V=900км/ч, м
					 7000.0, -- дальность ракурс 0(в догон) град, Н=10000м, V=900км/ч, м
					 13000.0, -- дальность ракурс 180(навстречу) град, Н=1000м,  V=900км/ч, м
					 2500.0, -- дальность ракурс 0(в догон) град, Н=1000м, V=900км/ч, м
					 2500.0, -- смещение назад отнсительно нуля задней точки зоны достижимости
					 0.4, -- процент гарантированной дальности от дальности в ППС на высоте 1км
					-0.014, -- производная процента гарантированной дальности в ППС по высоте
					0.5, -- Изменение коэффициентов наклона кривой в верхнюю и нижнюю полусферы от высоты носителя.
				},  
				
	controller = {
		boost_start = 0.0,
		march_start = 0.15,
	},
	
	boost = {
		impulse								= 250,
		fuel_mass							= 0,
		work_time							= 0,
		nozzle_position						= {{-1.9, 0, 0}},
		nozzle_orientationXYZ				= {{0.0, 0.0, 0.0}},
		nozzle_exit_area 					= 0.00724,
		tail_width							= 0.4,
		smoke_color							= {1.0, 1.0, 1.0},
		smoke_transparency					= 0.9,
		custom_smoke_dissipation_factor		= 0.2,	
	},
	
	march = {
		impulse								= 250,
		fuel_mass							= 40,
		work_time							= 2.8,
		nozzle_position						= {{-1.9, 0, 0}},
		nozzle_orientationXYZ				= {{0.0, 0.0, 0.0}},
		nozzle_exit_area 					= 0.00724,
		tail_width							= 0.4,
		smoke_color							= {1.0, 1.0, 1.0},
		smoke_transparency					= 0.9,
		custom_smoke_dissipation_factor		= 0.2,
	},
	
	fm = AIM_7_FM,
	
	proximity_fuze = {
		radius		= 9.1,
		arm_delay	= 4.0,
	},
	
	seeker = {
		delay		= 0.6,
		op_time		= 75,
		FOV			= math.rad(92),
		max_w_LOS	= math.rad(15),
		sens_near_dist			= 100,
		sens_far_dist			= 30000,
		ccm_k0					= 2,
		aim_sigma				= 8,
		height_error_k			= 110;
		height_error_max_vel	= 150;
		height_error_max_h		= 450;
		hoj						= 1,
	},

	autopilot = {
		x_channel_delay		= 0.5,
		delay				= 2.0,
		op_time				= 45,
		Kconv				= 3.0,
		Knv					= 0.02,
		Kd					= 0.4,
		Ki					= 0.15,
		Kout				= 1.0,
		Kx					= 0.1,
		Krx					= 2.0,
		fins_limit			= math.rad(20),
		fins_limit_x		= math.rad(10),
		Areq_limit			= 25.0,
		bang_bang			= 0,
		max_side_N			= 10,
		max_signal_Fi		= math.rad(12),
		rotate_fins_output	= 0,
		alg					= 0,
		PN_dist_data 		= {	15000,	0.5,
								4000,	0.5,
								2000,	1.0},
		null_roll			= math.rad(45),
	},
}

declare_weapon(AIM_7E)

local AIM_7E_2 =
{
	category		= CAT_AIR_TO_AIR,
	name			= "HB-AIM-7E-2",
	displayName		= _("AIM-7E-2 Sparrow Semi-Active Radar"),
	user_name		= _("AIM-7E-2"),
	scheme			= "aa_missile_semi_active",
	class_name		= "wAmmunitionSelfHoming",
	model			= "aim-7",
	wsTypeOfWeapon 	= {wsType_Weapon,wsType_Missile,wsType_AA_Missile,WSTYPE_PLACEHOLDER},
	
	mass			= 230,
	Escort			= 1,
	Head_Type		= 6,
	sigma			= {5.6, 5, 5.6},
	H_max			= 24400.0,
	H_min			= 1.0,
	Diam			= 203.0,
	Cx_pil			= 2.21,
	D_max			= 20000.0,
	D_min			= 1500.0,
	Head_Form		= 1,
	Life_Time		= 90.0,
	Nr_max			= 25,
	v_min			= 140.0,
	v_mid			= 500.0,
	Mach_max		= 3.2,
	t_b				= 0.6,
	t_acc			= 3.3,
	t_marsh			= 11.0,
	Range_max		= 50000.0,
	H_min_t			= 15.0,
	Fi_start		= 0.4,
	Fi_rak			= 3.14152,
	Fi_excort		= 0.8,
	Fi_search		= 0.1,
	OmViz_max		= 0.35,
	exhaust			= {0.78, 0.78, 0.78, 0.3};
	X_back			= -2.0,
	Y_back			= -0.0,
	Z_back			= 0.0, -- -0.1,
	Reflection		= 0.08,
	KillDistance	= 9.0,
	M				= 230,
	hoj = 1,
	
	ccm_k0 = 2.0,
	rad_correction = 0,
	height_error_k = 110, -- если проекция скорости цели на линию визирования меньше этого значения, появляется ошибка
	height_error_max_vel = 150, -- пропорциональный коэффициент
	height_error_max_h = 450, -- максимальная высота, где появляется ошибка
	
	warhead		= predefined_warhead("AIM_7E"),
	warhead_air = predefined_warhead("AIM_7E"),
	
	shape_table_data =
	{
		{
			name  = "hb_aim_7_E_2";
			file  = "HB_F-4E_AIM7E";
			life  = 1;
			fire  = { 0, 1};
			username = "HB-AIM-7E-2";
			index 	 = WSTYPE_PLACEHOLDER,
		},
	},
		
	ModelData = {   58 ,  -- model params count
					0.9 ,   -- characteristic square (характеристическая площадь)
			
					-- параметры зависимости Сx
					0.0125 , -- планка Сx0 на дозвуке ( M << 1)
					0.052 , -- высота пика волнового кризиса
					0.010 , -- крутизна фронта на подходе к волновому кризису
					0.002 , -- планка Cx0 на сверхзвуке ( M >> 1)
					0.5  , -- крутизна спада за волновым кризисом 
					1.2  , -- коэффициент отвала поляры
					
					-- параметры зависимости Cy
					2.20, -- планка Cya на дозвуке ( M << 1)
					1.05, -- планка Cya на сверхзвуке ( M >> 1)
					1.20, -- крутизна спада(фронта) за волновым кризисом  

					0.18, -- ~10 degrees Alfa_max  максимальный балансировачный угол, радианы
					0.00, --угловая скорость создаваймая моментом газовых рулей
						
					--	t_statr   t_b      t_accel  t_march   t_inertial   t_break  t_end
					-1.0,        -1.0,       2.8,     0.0,      0.0,		 0.0,    1.0e9,           -- time interval
					 0.0,         0.0, 	   14.28,     0.0,      0.0,         0.0,    0.0,           -- fuel flow rate in second kg/sec(секундный расход массы топлива кг/сек)
					 0.0,         0.0,   35000.0,     0.0,      0.0,         0.0,    0.0,           -- thrust
					
					 1.0e9, -- таймер самоликвидации, сек
					 75.0, -- время работы энергосистемы
					 0.0, -- абсалютеая высота самоликвидации, м
					 1.5, -- время задержки включения управленя, сек
					 1.0e9, -- дальность до цели в момент пуска, выше которой выполняется маневр набора высоты 
					 1.0e9, -- дальность до цели на трассе, менее которой начинается завершение маневра набора высоты (длжен быть больше чем предылущий параметр) 
					 0.0,  -- синус угла возвышения траектории набора горки
					 50.0, -- продольное ускорения взведения взрывателя
					 0.0, -- модуль скорости сообщаймый катапультным устройством, вышибным зарядом и тд
					 1.19, -- характристика системы САУ-РАКЕТА,  коэф фильтра второго порядка K0
					 1.0, -- характристика системы САУ-РАКЕТА,  коэф фильтра второго порядка K1
					 2.0, -- характристика системы САУ-РАКЕТА,  полоса пропускания контура управления
					 -- ЗРП. Данные для рассчета дальностей пуска (индикация на прицеле)
					 14.0, -- производная дальности по скорости носителя на высоте 1км, ППС
					 -19.0, -- производная дальности по скорости цели на высоте 1км, ЗПС
					 -2.2, -- производная по высоте производной дальности по скорости цели, ЗПС
					 19000, -- дальность ракурс 180(навстречу) град, Н=5000м, V=900км/ч, м
					 4800.0, -- дальность ракурс 0(в догон) град, Н=5000м, V=900км/ч, м
					 26000.0, -- дальность ракурс 180(навстречу) град, Н=10000м, V=900км/ч, м
					 7000.0, -- дальность ракурс 0(в догон) град, Н=10000м, V=900км/ч, м
					 13000.0, -- дальность ракурс 180(навстречу) град, Н=1000м,  V=900км/ч, м
					 2500.0, -- дальность ракурс 0(в догон) град, Н=1000м, V=900км/ч, м
					 2500.0, -- смещение назад отнсительно нуля задней точки зоны достижимости
					 0.4, -- процент гарантированной дальности от дальности в ППС на высоте 1км
					-0.014, -- производная процента гарантированной дальности в ППС по высоте
					0.5, -- Изменение коэффициентов наклона кривой в верхнюю и нижнюю полусферы от высоты носителя.
				},  
				
	controller = {
		boost_start = 0.0,
		march_start = 0.15,
	},
	
	boost = {
		impulse								= 250,
		fuel_mass							= 0,
		work_time							= 0,
		nozzle_position						= {{-1.9, 0, 0}},
		nozzle_orientationXYZ				= {{0.0, 0.0, 0.0}},
		nozzle_exit_area 					= 0.00724,
		tail_width							= 0.4,
		smoke_color							= {1.0, 1.0, 1.0},
		smoke_transparency					= 0.9,
		custom_smoke_dissipation_factor		= 0.2,	
	},
	
	march = {
		impulse								= 250,
		fuel_mass							= 40,
		work_time							= 2.8,
		nozzle_position						= {{-1.9, 0, 0}},
		nozzle_orientationXYZ				= {{0.0, 0.0, 0.0}},
		nozzle_exit_area 					= 0.00724,
		tail_width							= 0.4,
		smoke_color							= {1.0, 1.0, 1.0},
		smoke_transparency					= 0.9,
		custom_smoke_dissipation_factor		= 0.2,
	},
	
	fm = AIM_7_FM,
	
	proximity_fuze = {
		radius		= 9.1,
		arm_delay	= 2.9,
	},
	
	seeker = {
		delay		= 0.6,
		op_time		= 75,
		FOV			= math.rad(92),
		max_w_LOS	= math.rad(15),
		sens_near_dist			= 100,
		sens_far_dist			= 30000,
		ccm_k0					= 2,
		aim_sigma				= 8,
		height_error_k			= 110;
		height_error_max_vel	= 150;
		height_error_max_h		= 450;
		hoj						= 1,
	},

	autopilot = {
		x_channel_delay		= 0.5,
		delay				= 0.0,
		op_time				= 35,
		Kconv				= 5.0, -- PN Gain
		Knv					= 0.05, -- Proportional Gain -> To Desired G
		Kd					= 0.4, -- Derrivative Gain ->  To Desired G
		Ki					= 0.15, -- Integral Gain -> To Desired G
		Kout				= 1.0, --Transfer function H(s) = 1 / (s*s*T^2 + 2*T*E*s + 1)
		Kx					= 0.1, -- One of these is the cutoff freq
		Krx					= 2.0, -- Probably this one
		fins_limit			= math.rad(20),
		fins_limit_x		= math.rad(10),
		Areq_limit			= 25.0,
		bang_bang			= 0,
		max_side_N			= 10,
		max_signal_Fi		= math.rad(12),
		rotate_fins_output	= 0,
		alg					= 0,
		PN_dist_data 		= {	15000,	1,
								9000,	1},
		null_roll			= math.rad(45),
	},
}

declare_weapon(AIM_7E_2)

GBU_8 = {

	category		= CAT_BOMBS,
	name			= "HB_F4E_GBU_8_HOBOS",
	user_name		= "GBU-8",
	model 			= "HB_F-4E_EXT_GBU8",
	display_name_short 	= "GBU-8",
	displayName		= _("GBU-8 HOBOS - 2000 lb TV Guided Bomb"),
	wsTypeOfWeapon	= {wsType_Weapon, wsType_Bomb, wsType_Bomb_Guided, WSTYPE_PLACEHOLDER},
	scheme			= "AGM-62",
	class_name		= "wAmmunitionSelfHoming",
	type			= 1,
	mass			= 1027,
	hMin			= 25,
	hMax			= 2000,
	VyHold			= -50,
	Ag				= -5,
	Reflection = 0.2,
	add_attributes = {"Missiles"},

	shape_table_data = {{
		name 	 = "HB_F-4E_EXT_GBU8",
		file	 = "HB_F-4E_EXT_GBU8",
		username = "GBU-8",
		index	 = WSTYPE_PLACEHOLDER,
	}},


	fm = {
		A = 0.36,
		I = 1524.763968,
		L = 4.368,
		Ma = 0.85727626967319,
		Ma_x = 0.1,
		Mw = 4.6132861341704,
		Mw_x = 1,
		Sw = 0.6453888,
		caliber = 0.457,
		cx_coeff = { 1.037, 0.91, 0.6, 0.382, 1.34 },
		dCydA = { 0.14404178394221, 0.036 },
		finsTau = 0.1,
		mass = 1027,
		maxAoa = 0.1221730476396,
		release_rnd = 0.3,
		release_rnd_x = 1,
		wind_sigma = 0
	},
	
	seeker = {
		delay				= 0.0,
		op_time				= 800,
		activate_on_update	= 0,

		FOV					= math.rad(60),
		max_w_LOS			= 0.06,
		max_w_LOS_surf		= 0.03,

		max_target_speed			= 33,
		max_target_speed_rnd_coeff	= 10,

		max_lock_dist		= 60000,
	},
	
	PN_autopilot = {
		K			= 0.024,
		Ki			= 0.001,
		Kg			= 2.4,
		Kx			= 0.02,
		Krx			= 2,
		fins_limit	= 0.6,
		K_GBias		= 0.4,
	},
	
	march = {
		smoke_color			= {0.9, 0.9, 0.9},
		smoke_transparency	= 0.5,
	},
	
	arming_delay = {
		enabled = true,
		delay_time = 2
	},

	arming_vane = {
		enabled = false,
		velK = 1
	},

	targeting_data  = {
        char_time   = 20.3, -- characteristic time for sights
    },
	
	LaunchDistData =
	{
		24,		10,
	
				100,	150,	200,	250,	300,	350,	400,	450,	500,	550,		
		100,	500,	0,		2500,	5500,	9000,	12000,	14500,	17500,	20000,	23000,	
		200,	0,		0,		3000,	6000,	9000,	12000,	15000,	17500,	20500,	23500,	
		300,	0,		0,		3000,	6500,	9500,	12500,	15500,	18000,	21000,	24000,	
		400,	0,		1500,	3500,	6500,	10000,	13000,	16000,	18500,	21500,	24500,	
		500,	0,		0,		4000,	7000,	10000,	13500,	16500,	19000,	21500,	24500,	
		600,	0,		0,		4000,	7500,	10500,	14000,	16500,	19500,	22000,	25000,	
		700,	0,		2000,	4500,	7500,	11000,	14000,	17000,	19500,	22500,	25500,	
		800,	0,		2000,	5000,	8000,	11500,	14500,	17500,	20000,	23000,	26000,	
		900,	1500,	2500,	5000,	8500,	11500,	15000,	17500,	20500,	23500,	26500,	
		1000,	1500,	3000,	5500,	9000,	12000,	15000,	18000,	21000,	24000,	26500,	
		2000,	3000,	5000,	8500,	12000,	15500,	19000,	22000,	25000,	27500,	31000,	
		3000,	4000,	7500,	11500,	15000,	18500,	22500,	25500,	29000,	32000,	35000,	
		4000,	5500,	10500,	14000,	18000,	21500,	25500,	29000,	32500,	36000,	39500,	
		5000,	7000,	14000,	17000,	20500,	24500,	28500,	32500,	36500,	40000,	43500,	
		6000,	9000,	16500,	19500,	23500,	27500,	31500,	36000,	40000,	44000,	48000,	
		7000,	15500,	18500,	22000,	26000,	30000,	34500,	39000,	43500,	48000,	52000,	
		8000,	17500,	21000,	24500,	28500,	32500,	37000,	42000,	46500,	51500,	56000,	
		9000,	19000,	23000,	27000,	31000,	35500,	40000,	45000,	50000,	55000,	60000,	
		10000,	21000,	25500,	29500,	33500,	37500,	42500,	47500,	53000,	58500,	63000,	
		11000,	23000,	27500,	31500,	35500,	40000,	45000,	50500,	56500,	61500,	65500,	
		12000,	24500,	29500,	34000,	38000,	42500,	47500,	53500,	59000,	63500,	67500,	
		13000,	26000,	31500,	36000,	40000,	44500,	50000,	56000,	61000,	65000,	69000,	
		14000,	27500,	33000,	38000,	42500,	47000,	52500,	58500,	62500,	66500,	70500,	
		15000,	29000,	35000,	40000,	44500,	49000,	54500,	59500,	63000,	67500,	71500,	
	},

	MinLaunchDistData =
	{
		24,		10,
	
				100,	150,	200,	250,	300,	350,	400,	450,	500,	550,		
		100,	500,	800,	1150,	1600,	2100,	2650,	3200,	3300,	3600,	3850,	
		200,	700,	1100,	1500,	2000,	2550,	3100,	3550,	3700,	4250,	4450,	
		300,	850,	1300,	1800,	2300,	2850,	3400,	3950,	4100,	4600,	4900,	
		400,	0,		1500,	2000,	2550,	3100,	3700,	4150,	4450,	4950,	5300,	
		500,	1100,	1650,	2200,	2750,	3350,	3900,	4400,	4700,	5250,	5650,	
		600,	1200,	1750,	2350,	2950,	3550,	4150,	4600,	4950,	5500,	6050,	
		700,	1250,	1900,	2500,	3100,	3700,	4300,	4800,	5200,	5750,	6250,	
		800,	1350,	2000,	2650,	3250,	3900,	4500,	4950,	5400,	6000,	6550,	
		900,	1400,	2100,	2750,	3400,	4050,	4650,	5200,	5550,	6200,	6750,	
		1000,	1500,	2200,	2900,	3550,	4150,	4750,	5300,	5750,	6400,	7000,	
		2000,	1950,	2850,	3650,	4450,	5100,	5800,	6350,	7000,	7850,	8750,	
		3000,	2300,	3300,	4300,	5000,	5700,	6400,	7000,	7800,	8900,	10000,	
		4000,	2500,	3700,	4800,	5700,	6500,	7200,	7400,	8500,	9700,	10900,	
		5000,	2750,	4000,	5250,	6250,	7250,	8250,	9000,	9250,	10500,	11750,	
		6000,	3000,	4500,	5500,	7000,	8000,	9000,	10000,	11000,	11500,	12500,	
		7000,	3000,	4500,	6000,	7500,	8500,	10000,	11000,	12000,	12500,	13500,	
		8000,	3500,	4500,	6500,	8000,	9000,	10500,	12000,	13000,	14000,	14500,	
		9000,	3500,	5000,	6500,	8000,	9500,	11000,	12500,	14000,	15000,	16000,	
		10000,	3500,	5000,	6500,	8500,	10000,	12000,	13500,	15000,	16000,	17500,	
		11000,	4000,	5000,	7000,	9000,	10500,	12500,	14000,	16000,	17500,	18500,	
		12000,	4000,	5500,	7000,	9000,	11000,	13000,	15000,	16500,	18000,	19500,	
		13000,	4000,	5500,	7500,	9500,	11500,	13500,	15500,	17500,	19000,	20500,	
		14000,	4500,	5500,	7500,	10000,	12000,	14000,	16500,	18000,	20000,	21500,	
		15000,	4500,	6000,	8000,	10000,	12500,	14500,	17000,	19000,	21000,	22500,	
	},

    warhead	= predefined_warhead("Mk_84"),
}
declare_weapon(GBU_8)
declare_loadout({
	category = CAT_BOMBS,
	CLSID = "{HB_F4E_GBU_8}",
	attribute = GBU_8.wsTypeOfWeapon,
	Count = 1,
	--Cx_pil = 0.00194,
	Picture = "Picto_GBU8.png",
	displayName = GBU_8.displayName,
	Weight = GBU_8.mass,
	Elements = {{ ShapeName = GBU_8.model }},
})


--


HB_F4E_GBU15V1 = {
	category		= CAT_BOMBS,
	name			= "HB_F4E_GBU15V1",
	user_name		= "GBU-15-V1",
	displayName		= _("GBU-15 V1 - 2000 lb TV Guided Bomb"),
	wsTypeOfWeapon	= {wsType_Weapon, wsType_Bomb, wsType_Bomb_A, WSTYPE_PLACEHOLDER},
	scheme			= "bomb-common",
	model			= "HB_F4E_GBU15V1",
	type			= 1,
	mass			= 1136,
	hMin			= 25,
	hMax			= 2000,
	VyHold			= -50,
	Ag				= -5,

	fm = {
		mass		= 1136,
		caliber		= 0.457,
		cx_coeff	= { 1, 0.39, 0.6, 0.168, 1.31 },
		L			= 3.84,
		I			= 1 / 12 * 1136 * 3.84 ^ 2,
		Ma			= 2.746331,
		Mw			= 2.146083,
		wind_time	= 1000,
		wind_sigma	= 220,
	},

	arming_delay = {
		enabled = true,
		delay_time = 2
	},

	

	warhead = predefined_warhead("Mk_84"),

	shape_table_data = {{
		file	 = "HB_F4E_GBU15V1",
		username = "HB_F4E_GBU15V1",
		index	 = WSTYPE_PLACEHOLDER,
	}},

	targeting_data  = {
        char_time   = 20.60, -- characteristic time for sights
    },
}
declare_weapon(HB_F4E_GBU15V1)

declare_loadout({
	category = CAT_BOMBS,
	CLSID = "{GBU-15V1}",
	attribute = HB_F4E_GBU15V1.wsTypeOfWeapon,
	Count = 1,
	Cx_pil = 0.00194,
	Picture = "Picto_GBU15.png",
	displayName = HB_F4E_GBU15V1.displayName,
	Weight = HB_F4E_GBU15V1.mass,
	Elements = {{ ShapeName = HB_F4E_GBU15V1.model }},
})


local m61_20_hei = {
	category = CAT_SHELLS,
	name = "M56A3_HE_RED",
	user_name = _("M56A3 20mm HEI"),
	model_name    = "tracer_bullet_red",
	  projectile      = "HE",
	  payloadEffect   = "Thermal",
	  payloadMaterial = "RDX",
	  payload = 0.110,
	v0    = 3320*0.3048,
	Dv0   = 50.0/3320.0,
	Da0     = 0.001,
	Da1     = 0.0,
	mass      = 0.110,
	round_mass = 0.349,
	explosive     = 0.110, -- 0.0170 kg in real
	life_time     = 30,
	caliber     = 20.0,
	s         = 0.0,
	j         = 0.0,
	l         = 0.0,
	charTime    = 0,
	cx        = {0.5,1.27,0.70,0.200,2.30},
	k1        = 2.0e-08,

	tracer_off    = -1,
	tracer_on    = -1,
	scale_tracer  = 0,

	smoke_tail_life_time = 0.2,
	scale_smoke = 1,
	smoke_opacity = 0.1,
  
	cartridge = 0,
}
declare_weapon(m61_20_hei)
  
  local m61_20_api = {
	category = CAT_SHELLS,
	name = "M53_AP_RED",
	user_name = _("M53 20mm API"),
	model_name    = "tracer_bullet_red",
	  projectile      = "AP",
	v0    = 3380*0.3048,
	Dv0   = 50.0/3380.0,
	Da0     = 0.001,
	Da1     = 0.0,
	mass      = 0.110,
	round_mass = 0.349,
	explosive     = 0.0,
	life_time     = 30,
	caliber     = 20.0,
	s         = 0.0,
	j         = 0.0,
	l         = 0.0,
	charTime    = 0,
	cx        = {0.5,1.27,0.70,0.200,2.30},
	k1        = 2.0e-08,

	tracer_off    = -1,
	tracer_on    = -1,
	scale_tracer  = 0,

	smoke_tail_life_time = 0.2,
	scale_smoke = 1,
	smoke_opacity = 0.1,
  
	cartridge = 0,
}
declare_weapon(m61_20_api)
  
  -- M55A2  TP ball hollow
  local m61_20_tp = {
	category = CAT_SHELLS,
	name = "M55A2_TP_RED",
	user_name = _("M55A2 20mm TP"),
	model_name    = "tracer_bullet_red",
	  projectile      = "Ball",
	  payloadEffect   = "HollowCharge",
	v0    = 3380*0.3048,
	Dv0   = 50.0/3380.0,
	Da0     = 0.001,
	Da1     = 0.0,
	mass      = 0.110,
	round_mass = 0.349,
	explosive     = 0.0,
	life_time     = 30,
	caliber     = 20.0,
	s         = 0.0,
	j         = 0.0,
	l         = 0.0,
	charTime    = 0,
	cx        = {0.5,1.27,0.70,0.200,2.30},
	k1        = 2.0e-08,

	tracer_off    = -1,
	tracer_on    = -1,
	scale_tracer  = 0,

	smoke_tail_life_time = 0.2,
	scale_smoke = 1,
	smoke_opacity = 0.1,
  
	cartridge = 0,
}
declare_weapon(m61_20_tp)
  
  local m61_20_tracer = {
	category = CAT_SHELLS,
	name = "M53_APT_RED",
	user_name = _("20mm tracer"),
	model_name    = "tracer_bullet_red",
	  projectile      = "AP",
	v0    = 3380*0.3048,
	Dv0   = 50.0/3380.0,
	Da0     = 0.001,
	Da1     = 0.0,
	mass      = 0.110,
	round_mass = 0.349,
	explosive     = 0.0,
	life_time     = 30,
	caliber     = 20.0,
	s         = 0.0,
	j         = 0.0,
	l         = 0.0,
	charTime    = 0,
	cx        = {0.5,1.27,0.70,0.200,2.30},
	k1        = 2.0e-08,

	tracer_off = 6.1,
	tracer_on = 0.01,
	scale_tracer = 1,

	smoke_tail_life_time = 1.5,
	scale_smoke = 1,
	smoke_opacity = 0.1,
  
	cartridge = 0,
}
declare_weapon(m61_20_tracer)
-- TODO: PGU-28 (for F-14B, as mentioned in F14AAP-1) [slightly faster, less drag, e.g. https://www.gd-ots.com/munitions/medium-caliber-ammunition/20mm-x-102-pgu/]
  
local function m61a1(tbl)
	tbl.category = CAT_GUN_MOUNT
	tbl.name =  "M-61A1"
	tbl.display_name =  _("M-61A1")
	tbl.supply      =
	{
		shells = {"M56A3_HE_RED", "M53_AP_RED", "M55A2_TP_RED", "M53_APT_RED"},
		mixes  = { {1,1,1,1,4}, {2,2,2,2,4}, {1,2,1,2,4}, {3,3,3,3,4} },
		count  = 639, --639 according to Weapons Delivery Manual
	}
	if tbl.mixes then
		tbl.supply.mixes = tbl.mixes
		tbl.mixes        = nil
	end
	tbl.gun =
	{
		max_burst_length    = 100000,
		rates               = {6000,4000},
		recoil_coeff        = 0.7*1.3,
		barrels_count       = 1,
	}
	if tbl.rates then
		tbl.gun.rates        =  tbl.rates
		tbl.rates            = nil
	end

	tbl.ejector_pos             = tbl.ejector_pos or {0, 0, 0}
	tbl.ejector_pos_connector   = tbl.ejector_pos_connector     or  "Gun_point"
	tbl.ejector_dir             = tbl.ejector_dir or {0, 0, 0}
	tbl.supply_position         = tbl.supply_position   or {0,  0, 0}
	tbl.aft_gun_mount           = false
	tbl.effective_fire_distance = 2700
	tbl.drop_cartridge          = 0
	tbl.muzzle_pos              = tbl.muzzle_pos            or  {0,0,0}     -- all position from connector
	tbl.muzzle_pos_connector    = tbl.muzzle_pos_connector  or  "Gun_point" -- all position from connector
	tbl.azimuth_initial         = tbl.azimuth_initial       or  0
	tbl.elevation_initial       = tbl.elevation_initial     or  0
	tbl.smoke_dir               = tbl.smoke_dir             or  {0, 0, 0}
	if  tbl.effects == nil then
		tbl.effects = {{ name = "FireEffect", arg = tbl.effect_arg_number or 350 },{name = "SmokeEffect", smoke_exhaust = "PNT_GUN_SMOKE" , add_speed = {10,0,0}}}

		--tbl.effects = {{ name = "FireEffect"     , arg = tbl.effect_arg_number or 436 },
		--               { name = "HeatEffectExt"  , shot_heat = 7.823, barrel_k = 0.462 * 2.7, body_k = 0.462 * 14.3 },
		--               { name = "SmokeEffect"}}
	end
	return declare_weapon(tbl)
end

function make_ale40_config(flares,chaff)

	local name = "HB_ALE_40_"..flares.."_"..chaff

	local base_name
	
	local picture = "Picto_ALE40Disp.png"

	if (flares == 0 and chaff == 0) then
		base_name = "ALE-40 Dispensers (Empty)"
		picture = "Picto_ALE40Disp.png"
	elseif flares == 0 then
		base_name = "ALE-40 Dispensers ("..chaff.." Chaff)"
		picture = "Picto_ALE40Disp_120C.png"
	elseif chaff == 0 then
		base_name = "ALE-40 Dispensers ("..flares.." Flares)"
		picture = "Picto_ALE40Disp_30F.png"
	elseif flares == 30 and chaff == 60 then
		base_name = "ALE-40 Dispensers ("..flares.." Flares + "..chaff.." Chaff)"
		picture = "Picto_ALE40Disp_30F60C.png"
	elseif flares == 15 and chaff == 90 then
		base_name = "ALE-40 Dispensers ("..flares.." Flares + "..chaff.." Chaff)"
		picture = "Picto_ALE40Disp_15F90C.png"
	
	--else
	--	base_name = "ALE-40 Dispensers ("..flares.." Flares + "..chaff.." Chaff)"

	end

	local ale40 = {
		category = CAT_PODS,
		CLSID = "{"..name.."}",
		Picture = picture,
		displayName = _(base_name),
		Weight = 0, -- done on our side
		Cx_pil = 2.8*2 / 1.8 * 0.00035, -- master + slave (2 dispensers) drag index is 2.8
		attribute = { wsType_Weapon, wsType_GContainer, wsType_Control_Cont, WSTYPE_PLACEHOLDER },
		shape_table_data = {
			{
				name = name,
				file = "",
				username = name,
				life = 1,
				fire = {0,1},
				index = WSTYPE_PLACEHOLDER,
			},
		},
		Elements = {
			[1] = {
				ShapeName = name,
			},
		},
	}

	declare_loadout(ale40)

end

make_ale40_config(0, 0) -- {HB_ALE_40_0_0}
make_ale40_config(0, 120) -- {HB_ALE_40_0_120}
make_ale40_config(30, 60) -- {HB_ALE_40_30_60}
make_ale40_config(15, 90) -- {HB_ALE_40_15_90}
make_ale40_config(30, 0) -- {HB_ALE_40_30_0}

local suu_23 = {
category = CAT_PODS,
CLSID = "{SUU_23_POD}",
attribute       = {wsType_Weapon,wsType_GContainer,wsType_Cannon_Cont,WSTYPE_PLACEHOLDER},
wsTypeOfWeapon  = {wsType_Weapon,wsType_Shell,wsType_Shell,WSTYPE_PLACEHOLDER},
Picture         = "Picto_SUU23.png",
displayName     = _("SUU-23"),
Weight          = 612.35,      -- 1350lb/612.35kg loaded (incl. 201kg of ammunition)
Cx_pil          = 0.001220703125,
Elements        = {{ShapeName = "HB_ORD_SUU-23"}},
kind_of_shipping = 2,   -- SOLID_MUNITION
gun_mounts      = {
	m61a1({muzzle_pos_connector = "PNT_MUZZLE",
			smoke_dir = {0,-1,-0.2},
			elevation_initial=0.0,
			effects = {
				gatling_effect(351, 6),
				fire_effect(350, 0.02, 2, {0, 0, 0}),
				{name = "SmokeEffect", smoke_exhaust = "PNT_MUZZLE", add_speed = {5,0,0}},
			}

	})
},
shape_table_data = {{file = 'HB_ORD_SUU-23'; username = 'SUU-23'; index = WSTYPE_PLACEHOLDER;}}
}
declare_loadout(suu_23)

-- Drag calculated using ratio of Drag Index to known DCS drag value.
-- MK-83 Cx_pil = 0.00035
-- MK-83 index = 1.8
-- (Store index / MK-83 index) * MK-83 Cx_pil

local GALLON_TO_KG = 3.785 * 0.8
local fletcher_370_weight_empty = 139.7
local fletcher_370_Capacity = 370 * GALLON_TO_KG
local fletcher_370_weight = fletcher_370_Capacity + fletcher_370_weight_empty


--Left side
declare_loadout({
	category = CAT_FUEL_TANKS,
	CLSID = "{F4_SARGENT_TANK_370_GAL}",
	attribute = {wsType_Air,wsType_Free_Fall,wsType_FuelTank,WSTYPE_PLACEHOLDER},
	Picture = "Picto_370GalTank.png",
	displayName = _("Sargent Fletcher Fuel Tank 370 gallons"),
	Weight_Empty = 	fletcher_370_weight_empty, -- Kg
	Weight = 		fletcher_370_weight, -- Kg
	Capacity = 		fletcher_370_Capacity,
	Cx_pil = 0.00124,
	shape_table_data = {
		{
			name = "HB_F-4E_EXT_WingTank",
			file = "HB_F-4E_EXT_WingTank",
			username = "HB_F-4E_EXT_WingTank",
			life = 1,
			fire = {0,1},
			index = WSTYPE_PLACEHOLDER,
		},
	},
	Elements = {
		{
			ShapeName = "HB_F-4E_EXT_WingTank",
		},
	}
})



--Right wing tank. The left and right need to differ because of liveries unfortunately.
declare_loadout({
	category = CAT_FUEL_TANKS,
	CLSID = "{F4_SARGENT_TANK_370_GAL_R}",
	attribute = {wsType_Air,wsType_Free_Fall,wsType_FuelTank,WSTYPE_PLACEHOLDER},
	Picture = "Picto_370GalTank.png",
	displayName = _("Sargent Fletcher Fuel Tank 370 gallons"),
	Weight_Empty = 	fletcher_370_weight_empty, -- Kg
	Weight = 		fletcher_370_weight, -- Kg
	Capacity = 		fletcher_370_Capacity,
	Cx_pil = 0.00124,
	shape_table_data = {
		{
			name = "HB_F-4E_EXT_WingTank_R",
			file = "HB_F-4E_EXT_WingTank_R",
			username = "HB_F-4E_EXT_WingTank_R",
			life = 1,
			fire = {0,1},
			index = WSTYPE_PLACEHOLDER,
		},
	},
	Elements = {
		{
			ShapeName = "HB_F-4E_EXT_WingTank_R",
		},
	}
})

local fletcher_600_weight_empty = 112.9
local fletcher_600_Capacity = 600 * GALLON_TO_KG
local fletcher_600_weight = fletcher_600_Capacity + fletcher_600_weight_empty

-- Weights and Drags used for the MCDONNELL Tank seen in the F-4J natops.
declare_loadout({
	category = CAT_FUEL_TANKS,
	CLSID = "{F4_SARGENT_TANK_600_GAL}",
	attribute = {wsType_Air,wsType_Free_Fall,wsType_FuelTank,WSTYPE_PLACEHOLDER},
	Picture = "Picto_600GalTank.png",
	displayName = _("Sargent Fletcher Fuel Tank 600 gallons"),
	Weight = 		fletcher_600_weight, -- Kg
	Weight_Empty = 	fletcher_600_weight_empty, -- Kg
	Capacity = 		fletcher_600_Capacity,
	Cx_pil = 0.00187,
	shape_table_data = {
		{
			name = "HB_F-4E_EXT_Center_Fuel_Tank",
			file = "HB_F-4E_EXT_Center_Fuel_Tank",
			username = "HB_F-4E_EXT_Center_Fuel_Tank",
			life = 1,
			fire = {0,1},
			index = WSTYPE_PLACEHOLDER,
		},
	},
	Elements = {
		{
			ShapeName = "HB_F-4E_EXT_Center_Fuel_Tank",
		},
	}
})

local hp_600_weight_empty = 153.3
local hp_600_Capacity = 600 * GALLON_TO_KG
local hp_600_weight = hp_600_Capacity + hp_600_weight_empty

declare_loadout({
	category = CAT_FUEL_TANKS,
	CLSID = "{F4_HIGH_PERFORMANCE_CENTERLINE_600_GAL}",
	attribute = {wsType_Air,wsType_Free_Fall,wsType_FuelTank,WSTYPE_PLACEHOLDER},
	Picture = "AKANM55.png", --todo placeholder
	displayName = _("High Performance Centerline Tank 600 gallons"),
	Weight = 			hp_600_weight, -- Kg
	Weight_Empty = 		hp_600_weight_empty, -- Kg
	Capacity = 			hp_600_Capacity,
	Cx_pil = 0.00187,
	shape_table_data = {
		{
			name = "HB_HIGH_PERFORMANCE_CENTERLINE_600_GAL",
			file = "HB_HIGH_PERFORMANCE_CENTERLINE_600_GAL",
			username = "HB_HIGH_PERFORMANCE_CENTERLINE_600_GAL",
			life = 1,
			fire = {0,1},
			index = WSTYPE_PLACEHOLDER,
		},
	},
	Elements = {
		{
			ShapeName = "HB_HIGH_PERFORMANCE_CENTERLINE_600_GAL",
		},
	}
})



---------------------------- EMPTY TANKS ----------------------------
--Left side
declare_loadout({
	category = CAT_FUEL_TANKS,
	CLSID = "{F4_SARGENT_TANK_370_GAL_EMPTY}",
	attribute = {wsType_Air,wsType_Free_Fall,wsType_FuelTank,WSTYPE_PLACEHOLDER},
	Picture = "Picto_370GalTank.png",
	displayName = _("Sargent Fletcher Fuel Tank 370 gallons (empty)"),
	Weight_Empty = 	fletcher_370_weight_empty, -- Kg
	Weight = 		fletcher_370_weight_empty + 10, -- Kg
	Capacity = 		fletcher_370_Capacity,
	Cx_pil = 0.00124,
	shape_table_data = {
		{
			name = "HB_F-4E_EXT_WingTank_EMPTY",
			file = "HB_F-4E_EXT_WingTank",
			username = "HB_F-4E_EXT_WingTank",
			life = 1,
			fire = {0,1},
			index = WSTYPE_PLACEHOLDER,
		},
	},
	Elements = {
		{
			ShapeName = "HB_F-4E_EXT_WingTank_EMPTY",
		},
	}
})



--Right wing tank. The left and right need to differ because of liveries unfortunately.
declare_loadout({
	category = CAT_FUEL_TANKS,
	CLSID = "{F4_SARGENT_TANK_370_GAL_R_EMPTY}",
	attribute = {wsType_Air,wsType_Free_Fall,wsType_FuelTank,WSTYPE_PLACEHOLDER},
	Picture = "Picto_370GalTank.png",
	displayName = _("Sargent Fletcher Fuel Tank 370 gallons (empty)"),
	Weight_Empty = 	fletcher_370_weight_empty, -- Kg
	Weight = 		fletcher_370_weight_empty + 10, -- Kg
	Capacity = 		fletcher_370_Capacity,
	Cx_pil = 0.00124,
	shape_table_data = {
		{
			name = "HB_F-4E_EXT_WingTank_R_EMPTY",
			file = "HB_F-4E_EXT_WingTank_R",
			username = "HB_F-4E_EXT_WingTank_R",
			life = 1,
			fire = {0,1},
			index = WSTYPE_PLACEHOLDER,
		},
	},
	Elements = {
		{
			ShapeName = "HB_F-4E_EXT_WingTank_R_EMPTY",
		},
	}
})

local fletcher_600_weight_empty = 112.9
local fletcher_600_Capacity = 600 * GALLON_TO_KG
local fletcher_600_weight = fletcher_600_Capacity + fletcher_600_weight_empty

-- Weights and Drags used for the MCDONNELL Tank seen in the F-4J natops.
declare_loadout({
	category = CAT_FUEL_TANKS,
	CLSID = "{F4_SARGENT_TANK_600_GAL_EMPTY}",
	attribute = {wsType_Air,wsType_Free_Fall,wsType_FuelTank,WSTYPE_PLACEHOLDER},
	Picture = "Picto_600GalTank.png",
	displayName = _("Sargent Fletcher Fuel Tank 600 gallons (empty)"),
	Weight = 		fletcher_370_weight_empty + 10, -- Kg
	Weight_Empty = 	fletcher_600_weight_empty, -- Kg
	Capacity = 		fletcher_600_Capacity,
	Cx_pil = 0.00187,
	shape_table_data = {
		{
			name = "HB_F-4E_EXT_Center_Fuel_Tank_EMPTY",
			file = "HB_F-4E_EXT_Center_Fuel_Tank",
			username = "HB_F-4E_EXT_Center_Fuel_Tank",
			life = 1,
			fire = {0,1},
			index = WSTYPE_PLACEHOLDER,
		},
	},
	Elements = {
		{
			ShapeName = "HB_F-4E_EXT_Center_Fuel_Tank_EMPTY",
		},
	}
})


-- TODO waiting for model.
--declare_loadout(high_performance_centerline_tank)

local aim9_variants =
{
	["AIM-9"]  		= {display_name = _("AIM-9M")	  	 			,wstype = {4,	4,	7	, AIM_9 },	category = CAT_AIR_TO_AIR, mass = 86.64	},
	["AIM-9P"] 		= {display_name = _("AIM-9P")	  	 			,wstype = {4,	4,	7	, AIM_9P},	category = CAT_AIR_TO_AIR, mass = 86.18	},
	["AIM-9L"]		= {display_name = _("AIM-9L")					,wstype = "weapons.missiles.AIM-9L",	category = CAT_AIR_TO_AIR	},
}

local bomb_data =
{
	["MK-81"]			= {name = _("Mk-81 - 250lb GP Bomb LD"),						mass = 118, 	wsType = {4, 5, 9, 30},  												Cx = 0.00013, picture = "FAB100.png"	, 	settings = Get_Combined_GUISettings_Preset("MDRN_B_A_GPLD")	},
	["MK-82"]			= {name = _("Mk-82 - 500lb GP Bomb LD"),						mass = 228, 	wsType = {4, 5, 9, 31},  												Cx = 0.00025, picture = "mk82.png"		, 	settings = Get_Combined_GUISettings_Preset("MDRN_B_A_GPLD")	},
	["MK-82_Snakeye"]	= {name = _("Mk-82 Snakeye - 500lb GP Bomb HD"),				mass = 249.5, 	wsType = {4, 5, 9, 79},  												Cx = 0.00035, picture = "mk82AIR.png"	, 	settings = Get_Combined_GUISettings_Preset("MDRN_B_A_GPLD")	},
	["MK-82Y"]			= {name = _("Mk-82Y - 500lb GP Chute Retarded HD"), 			mass = 232, 	wsType = "weapons.bombs.Mk_82Y",  										Cx = 0.00035, picture = "mk82AIR.png"	, 	settings = Get_Fuze_GUISettings_Preset("MDRN_B_A_TPHD")		},
	["MK-82AIR"]		= {name = _("Mk-82 AIR Ballute - 500lb GP Bomb HD"), 			mass = 242, 	wsType = {4, 5, 9, 75},  												Cx = 0.00035, picture = "mk82AIR.png"	, 	settings = Get_Combined_GUISettings_Preset("MDRN_B_A_GPLD")	},
	["ROCKEYE"]			= {name = _("Mk-20 Rockeye - 490lbs CBU, 247 x HEAT Bomblets"),	mass = 222, 	wsType = {4, 5, 38, 45}, 												Cx = 0.000413, picture = "Mk20.png"		, 	settings = Get_Fuze_GUISettings_Preset("MDRN_CC_A_Mk20")	},
	["BDU-33"]			= {name = _("BDU-33 - 25lb Practice Bomb LD"),					mass = 11.3, 	wsType = {4, 5, 9, 69}, 												Cx = 0.00025, picture = "bdu-33.png"																	},
	["MK-83"] 			= {name = _("Mk-83 - 1000lb GP Bomb LD"), 						mass = 454, 	wsType = {4, 5, 9, 32}, 												Cx = 0.00035, picture = "mk83.png"		, 	settings = Get_Combined_GUISettings_Preset("MDRN_B_A_GPLD")	},
	["GBU-12"] 			= {name = _("GBU-12 - 500lb Laser Guided Bomb"), 				mass = 277, 	wsType = {4, 5, 36, 38}, 												Cx = 0.000413, picture = "GBU12.png"	, 	settings = Get_Combined_GUISettings_Preset("Paveway_II")	},
	["GBU-16"] 			= {name = _("GBU-16 - 1000lb Laser Guided Bomb"), 				mass = 513, 	wsType = {4, 5, 36, 39}, 												Cx = 0.000413, picture = "GBU16.png"	, 	settings = Get_Combined_GUISettings_Preset("Paveway_II")	},
	["BDU-45"]			= {name = _("BDU-45 - 500lb Practice Bomb"),					mass = 232, 	wsType = "weapons.bombs.BDU_45",  										Cx = 0.00035, picture = "mk82AIR.png"	, 	settings = Get_Combined_GUISettings_Preset("MDRN_B_A_GPLD")	},
	["BDU-45B"]			= {name = _("BDU-45B - 500lb Practice Bomb"),					mass = 232, 	wsType = "weapons.bombs.BDU_45B",  										Cx = 0.00025, picture = "mk82.png"		, 	settings = Get_Combined_GUISettings_Preset("MDRN_B_A_GPLD")	},
	["BDU-50LD"]  		= {name = _("BDU-50LD - 500lb Practice Bomb LD"), 				mass = 241, 	wsType = {wsType_Weapon,	wsType_Bomb,	wsType_Bomb_A,	BDU_50LD}, 	Cx = 0.00018, picture = 'bdu-50ld.png',		settings = Get_Fuze_GUISettings_Preset("MDRN_B_A_TPLD") 	},
	["BDU-50HD"]  		= {name = _("BDU-50HD - 500lb Practice Bomb HD"), 				mass = 241, 	wsType = {wsType_Weapon,	wsType_Bomb,	wsType_Bomb_A,	BDU_50HD}, 	Cx = 0.00018, picture = 'bdu-50hd.png',		settings = Get_Fuze_GUISettings_Preset("MDRN_B_A_TPLD") 	},
	["CBU-87"]			= {name = _("CBU-87 - 202 x CEM Cluster Bomb"),	 				mass = 430, 	wsType = {4,	5,	38,	77}, 											Cx = 0.000413, picture = "CBU.png",			settings = Get_Combined_GUISettings_Preset("CBU_SUU65Plus")	},
	["BDU_45LGB"]		= {name = _("BDU-45 LG - 500lb Practice Laser Guided Bomb"), 	mass = 277, 	wsType = "weapons.bombs.BDU_45LGB", 									Cx = 0.000365, picture = "GBU12.png", 		settings = Get_LGU_GUISettings_Preset("MAU169")				},
	["M117"]			= {name = _("M117 - 750lb GP Bomb LD"), 						mass = 340, 	wsType = {4,5,9,34}, 													Cx = 0.001, picture = "KMGU2.png" 																		},
	["CBU-52B"]			= {name = _("CBU-52B - 220 x HE/Frag bomblets"), 				mass = 356, 	wsType = {4,5,38,93}, 													Cx = 0.000265, picture = "CBU.png"		},
	["BLU-107B"]		= {name = _("BLU-107/B Durandal - 219kg Concrete Piercing Chute Retarded Bomb w/Booster"), mass = 219, wsType = "weapons.bombs.Durandal", 				Cx = 0.0006, picture = "blu107.png"		},

}

-- A note on the Mavericks below; the A and the B have fully defined WSTypes, but the D, for whatever reason, only has the final WSType digit defined.
local missile_data =
{
	["AGM-65A"]		= {name = "AGM-65A - Maverick A (TV Guided)", mass = 210.5, wstype = "weapons.missiles.AGM_65A", pic = 'agm65.png',	cx = 0.00097, offset = 0.07, PictureBlendColor = false },
	["AGM-65B"]		= {name = "AGM-65B - Maverick B (TV Guided)", mass = 210.5, wstype = "weapons.missiles.AGM_65B", pic = 'agm65.png',	cx = 0.00097, offset = 0.07, PictureBlendColor = false },
	["AGM-65D"]		= {name = "AGM-65D - Maverick D (IIR ASM)", mass = 210.5, wstype = {4, 4, 8, AGM_65D }, pic = 'agm65.png',	cx = 0.00097, offset = 0.07, PictureBlendColor = false },
	["AGM-65G"]		= {name = "AGM-65G - Maverick G (IIR ASM - Lg Whd)", mass = 301, wstype = "weapons.missiles.AGM_65G", pic = 'agm65.png',	cx = 0.00097, offset = 0.07, PictureBlendColor = false },
}

local rocket_data = {	
	["LAU-3_WP156"]		= {name = "LAU-3 pod - 19 x 2.75\" FFAR, UnGd Rkts M156, Wht Phos",	CLSID = "{LAU3_FFAR_WP156}",	mass = 247.4, 	mass_empty = 34.5, 	wsType = {4,7,33,148}, Cx = 0.00146484375, picture = "LAU61.png"},
	["LAU-3_MK1"]		= {name = "LAU-3 pod - 19 x 2.75\" FFAR, UnGd Rkts Mk1, HE",	CLSID = "{LAU3_FFAR_MK1HE}",	mass = 219.8, 		mass_empty = 34.5, 	wsType = {4,7,33,144}, Cx = 0.00146484375, picture = "LAU61.png"},
	["LAU-3_MK5"]		= {name = "LAU-3 pod - 19 x 2.75\" FFAR, UnGd Rkts Mk5, HEAT",	CLSID = "{LAU3_FFAR_MK5HEAT}",	mass = 220.7, 		mass_empty = 34.5, 	wsType = {4,7,33,145}, Cx = 0.00146484375, picture = "LAU61.png"},

	-- SAME DRAG AS THE LAU-3 for some reason.
	["LAU-68_WP156"]	= {name = "LAU-68 pod - 7 x 2.75\" FFAR, UnGd Rkts M156, Wht Phos",	CLSID = "{LAU68_FFAR_WP156}",	mass = 120.2, 	mass_empty = 41.73, wsType = {4,7,33,148}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_MK1"]		= {name = "LAU-68 pod - 7 x 2.75\" FFAR, UnGd Rkts Mk1, HE",	CLSID = "{LAU68_FFAR_MK1HE}",	mass = 110.0, 		mass_empty = 41.73, wsType = {4,7,33,144}, Cx = 0.00146484375, picture = "LAU68.png"},
	["LAU-68_MK5"]		= {name = "LAU-68 pod - 7 x 2.75\" FFAR, UnGd Rkts Mk5, HEAT",	CLSID = "{LAU68_FFAR_MK5HEAT}",	mass = 110.3, 		mass_empty = 41.73, wsType = {4,7,33,145}, Cx = 0.00146484375, picture = "LAU68.png"},
}

local dispenser_data = {
	["SUU-25"]			= {name = "SUU-25 x 8 LUU-2 - Target Marker Flares",		CLSID = "{CAE48299-A294-4bad-8EE6-89EFC5DCDF00}",	mass = 226.8, 			mass_empty = 118, 					wsType = {wsType_Weapon, wsType_Bomb, wsType_Bomb_Lighter, LUU_2B}, Cx = 0.0005, 		picture = "L005.png"},
	["CBU-1A"]			= {name = "CBU-1A/A x 27x19 (513) BLU-4B Bomblets, HE", 	CLSID = "{HB_F4E_CBU-1/A}", 						mass = CBU_1A.Weight, 	mass_empty = CBU_1A.Weight_Empty, 	wsType = CBU_1A.wsTypeOfWeapon, 									Cx = CBU_1A.Cx_pil, picture= "Picto_CBU-1-2.png"},
	["CBU-2A"]			= {name = "CBU-2/A x 19x19 (361) BLU-3 Bomblets, HE", 		CLSID = "{HB_F4E_CBU-2/A}", 						mass = CBU_2A.Weight, 	mass_empty = CBU_2A.Weight_Empty, 	wsType = CBU_2A.wsTypeOfWeapon, 									Cx = CBU_2A.Cx_pil, picture= "Picto_CBU-1-2.png"},
	["CBU-2BA"]			= {name = "CBU-2B/A x 22x19 (418) BLU-3B Bomblets, HE", 	CLSID = "{HB_F4E_CBU-2B/A}", 						mass = CBU_2BA.Weight, 	mass_empty = CBU_2BA.Weight_Empty, 	wsType = CBU_2BA.wsTypeOfWeapon, 									Cx = CBU_2BA.Cx_pil, picture= "Picto_CBU-1-2.png"},
}

local function change_category(dispenser)

	local data = dispenser_data[dispenser]

	return {
			category			= CAT_PODS,
			CLSID				= "{HB_SUU-25}",
			attribute			= data.wsType,
			Count				= 1,
			Cx_pil				= data.Cx,
			Picture				= data.picture,
			displayName			= data.name,
			Weight				= data.mass,
			Weight_Empty 		= data.mass_empty,
			Elements = {{ payload_CLSID	= data.CLSID	}},
			settings			= Get_Fuze_GUISettings_Preset("MDRN_B_A_GPHD")
	}
end

-- This just declutters the bombs and puts them in the pods section
--declare_loadout(change_category("SUU-25")) -- Doesn't work for now need to use the correct model.

local function aim_9_with_adapter(CLSID,aim_9_variant,start_from_side)
	local var 	   = aim9_variants[aim_9_variant] or aim9_variants["AIM-9"]
	local var_mass = var.mass or 85.5
	local name_prefix = "2x "

	side1 = "Left"
	side2 = "Right"
	if (start_from_side == "Right") then
		side1 = "Right"
		side2 = "Left"
	end

	declare_loadout({
		category			= var.category,
		CLSID 				= CLSID,
		Picture				=	"aim9p.png",
		displayName			=	name_prefix..var.display_name,
		wsTypeOfWeapon		=   var.wstype,
		attribute			=	{4,	4,	32,	111},
		Cx_pil				=	1.2*(2.58 / 4096.0),  -- 2.58 from AIM_9L.Cx_pil
		Count				=	2,
		Weight				=	15 + var_mass,
		JettisonSubmunitionOnly = true,

		Elements			=
		{
			{	ShapeName	=	"F4E_dual_LAU7"	   	  ,	IsAdapter  	   =   true},
			{	ShapeName	=	aim_9_variant	  ,	connector_name =  "PNT_AIM9_"..side1},
			{	ShapeName	=	aim_9_variant	  ,	connector_name =  "PNT_AIM9_"..side2},
		}-- end of Elements
	})
end

local function aim_9_without_adapter(CLSID,aim_9_variant)
	local var = aim9_variants[aim_9_variant] or aim9_variants["AIM-9"]
	local var_mass = var.mass or 85.5
	declare_loadout({
		category			= 	var.category,
		CLSID 				= 	CLSID,
		Picture				=	"aim9p.png",
		displayName			=	var.display_name,
		attribute			=	var.wstype,
		Cx_pil				=	2.58 / 4096.0,  -- 2.58 from AIM_9L.Cx_pil
		Count				=	1,
		Weight				=	var_mass,
		Elements			=  {{ShapeName = aim_9_variant}}-- end of Elements
	})
end

--for i,v in ipairs({"L","R"}) do
--end
--aim_9_with_adapter("{LAU-7 - AIM-9P}","AIM-9P")
aim_9_with_adapter("{LAU-7_AIM-9M_Left}","AIM-9", "Left")
aim_9_with_adapter("{LAU-7_AIM-9M_Right}","AIM-9", "Right")
aim_9_with_adapter("{LAU-7_AIM-9L_Left}","AIM-9L", "Left")
aim_9_with_adapter("{LAU-7_AIM-9L_Right}","AIM-9L", "Right")
aim_9_without_adapter("{AIM-9M}", "AIM-9")

function single_swa(bomb)

	local bomb_variant = bomb_data[bomb] or bomb_data["Mk-82"]

	return {
		category			= CAT_BOMBS,
		CLSID				= "{HB_"..bomb.."_SWA}",
		attribute			= bomb_variant.wsType,
		Count				= 1,
		Cx_pil				= bomb_variant.Cx,
		Picture				= bomb_variant.picture,
		displayName			= "(Special Weapons Adapter) "..bomb_variant.name,
		Weight				= bomb_variant.mass,
		Elements = {{ payload_CLSID	=	bomb_variant.CLSID	}},
		settings			= bomb_variant.settings,
	}
end


-- FROM BEHIND
-- 4-----6  FWD
--  \   /
--    2
-- 3-----5  AFT
--  \   /
--    1
--
-- Example Config
-- {1,1,1, 1,1,1} -- Full
-- {1,1,1, 0,0,0} -- Full Front Empty Rear
-- {1,0,0, 1,0,0} -- Only Left Bombs
-- {0,1,0, 0,1,0} -- Only Centre Bombs
-- {0,0,1, 0,0,1} -- Only Right Bombs
function mer_config(config)

	local result = {}

	local conversion = {4,2,6,3,1,5}

	for i,v in ipairs(config) do
		if v == 1 then
			result[#result+1] = conversion[i]
		end
	end

	-- Sort result so drop order is correct.
	-- Last bomb on is fired first.
	table.sort(result, function(a,b) return a > b end )

	return result
end

local mer_all = {1,1,1, 1,1,1}

-- The following function creates a cluster of elements on a BRU-41 MER rack. Can fit up to 6x.
-- Valid options for "side" are left, right and center.
local function mer(element, clsid, config, comment)
	config = config or {1,1,1, 1,1,1}
	config = mer_config(config)

	comment = comment or ""

	local count = #config
	local bomb_variant = bomb_data[element] or bomb_data["Mk-82"]
	local rack_variant = {mass = 99.8, shapename = "HB_ORD_MER", wstype = {4, 5, 32, WSTYPE_PLACEHOLDER}, Cx = 0.0024 }
	local populated_rack = {}

	populated_rack.category           = CAT_BOMBS
	populated_rack.CLSID              = clsid
	populated_rack.attribute          = rack_variant.wstype
	populated_rack.Picture            = bomb_variant.picture
	if bomb_variant.PictureBlendColor then
		populated_rack.PictureBlendColor  = "0xffffffff"
	end
	populated_rack.Count              = count
	populated_rack.displayName        = tostring(count).."x "..bomb_variant.name.." (MER)"..comment
	populated_rack.wsTypeOfWeapon     = bomb_variant.wsType
	populated_rack.Weight             = rack_variant.mass + count * bomb_variant.mass
	populated_rack.Cx_pil             = (bomb_variant.Cx * count) + rack_variant.Cx

	if type(bomb_variant.wsType)=="table" and bomb_variant.wsType[4] ~= nil then
		populated_rack.wsTypeOfWeapon = bomb_variant.wsType
	elseif type(bomb_variant.wsType)=="string" then
		populated_rack.wsTypeOfWeapon = bomb_variant.wsType
	end

	-- Now add the MER model:
	populated_rack.Elements = {
		{
			DrawArgs	=	{{1,0}},
			Position	=	{0,	0,	0},
			ShapeName	=	rack_variant.shapename,
			IsAdapter   =   true,
		},
	}

	local connector_names = {
		"POINT_PYLON_06",
		"POINT_PYLON_01",
		"POINT_PYLON_04",
		"POINT_PYLON_02",
		"POINT_PYLON_05",
		"POINT_PYLON_03",
	}

	local translate_fwd  = 1.091
	local translate_back = -1.177
	local offset = -0.0762

	local positions = {
		{ offset + translate_back, -0.309, 0 }, -- back center
		{ offset + translate_fwd, -0.309, 0 }, -- front center
		{ offset + translate_back, -0.112, -0.13 }, -- back left
		{ offset + translate_fwd, -0.112, -0.13 }, -- front left
		{ offset + translate_back, -0.112, 0.13 }, -- back right
		{ offset + translate_fwd, -0.112, 0.13 } -- front right
	}

	local rotations =  {
		{ 0,  0, 0 }, -- center
		{ 0,  0, 0 },
		{ 45, 0, 0 }, -- left side
		{ 45, 0, 0 },
		{ -45, 0, 0 }, -- right side
		{ -45, 0, 0 }
	}

	for i,v in ipairs(config) do
		populated_rack.Elements[#populated_rack.Elements + 1] = {DrawArgs	=	{{1,1},{2,1}},
											 --Position	=	positions[v],
											 ShapeName	=	element,
											 --Rotation	=   rotations[v],
											 connector_name = connector_names[v]
											 --payload_CLSID	=	bomb_variant.CLSID
		}
	end

	populated_rack.settings = bomb_variant.settings

	return populated_rack
end

local function mer_rocket(element, clsid, config, comment)
	config = config or {1,1,1, 1,1,1}
	config = mer_config(config)

	comment = comment or ""

	local count = #config
	local rocket_variant = rocket_data[element] or rocket_data["LAU-3_MK5"]
	local rack_variant = {mass = 99.8, shapename = "HB_ORD_MER", wstype = {4, 7, 32, WSTYPE_PLACEHOLDER}, Cx = 0.0024 }
	local populated_rack = {}

	populated_rack.category           = CAT_ROCKETS
	populated_rack.CLSID              = clsid
	populated_rack.attribute          = rack_variant.wstype
	populated_rack.Picture            = rocket_variant.picture
	if rocket_variant.PictureBlendColor then
		populated_rack.PictureBlendColor  = "0xffffffff"
	end
	populated_rack.Count              = count
	populated_rack.displayName        = tostring(count).."x "..rocket_variant.name.." (MER)"..comment
	populated_rack.wsTypeOfWeapon     = rocket_variant.wsType
	populated_rack.Weight             = rack_variant.mass + count * rocket_variant.mass
	populated_rack.Weight_Empty        = rack_variant.mass + count * rocket_variant.mass_empty
	populated_rack.Cx_pil             = (rocket_variant.Cx * count) + rack_variant.Cx

	if type(rocket_variant.wsType)=="table" and rocket_variant.wsType[4] ~= nil then
		populated_rack.wsTypeOfWeapon = rocket_variant.wsType
	elseif type(rocket_variant.wsType)=="string" then
		populated_rack.wsTypeOfWeapon = rocket_variant.wsType
	end

	-- Now add the MER model:
	populated_rack.Elements           =
	{
		{
			DrawArgs	=	{{1,0}},
			Position	=	{0,	0,	0},
			ShapeName	=	rack_variant.shapename,
			IsAdapter   =   true,
		},
	}

	local connector_names = {
		"POINT_PYLON_06",
		"POINT_PYLON_01",
		"POINT_PYLON_04",
		"POINT_PYLON_02",
		"POINT_PYLON_05",
		"POINT_PYLON_03",
	}

	for i,v in ipairs(config) do
		populated_rack.Elements[#populated_rack.Elements + 1] = {DrawArgs	=	{{1,1},{2,1}},
											 --Position	=	positions[v],
											 ShapeName	=	element,
											 connector_name = connector_names[v],
											 --Rotation	=   rotations[v],
											 payload_CLSID	=	rocket_variant.CLSID
											}
	end

	return populated_rack
end

local function mer_dispenser(element, clsid, config, comment)
	config = config or {1,1,1, 1,1,1}
	config = mer_config(config)

	comment = comment or ""

	local count = #config
	local rocket_variant = dispenser_data[element] or dispenser_data["SUU-25"]
	local rack_variant = {
		mass = 99.8, 
		shapename = "HB_ORD_MER", 
		wstype = {wsType_Weapon, wsType_Bomb, wsType_Container, WSTYPE_PLACEHOLDER}, 
		Cx = 0.0024 
	}
	local populated_rack = {}
	populated_rack.category           = CAT_PODS
	populated_rack.CLSID              = clsid
	populated_rack.attribute          = rack_variant.wstype
	populated_rack.Picture            = rocket_variant.picture
	if rocket_variant.PictureBlendColor then
		populated_rack.PictureBlendColor  = "0xffffffff"
	end
	populated_rack.Count              = count
	populated_rack.displayName        = tostring(count).."x "..rocket_variant.name.." (MER)"..comment
	populated_rack.wsTypeOfWeapon     = rocket_variant.wsType
	populated_rack.Weight             = rack_variant.mass + count * rocket_variant.mass
	populated_rack.Weight_Empty        = rack_variant.mass + count * rocket_variant.mass_empty
	populated_rack.Cx_pil             = (rocket_variant.Cx * count) + rack_variant.Cx

	populated_rack.ejectDirection = {-1, 0, 0}
	populated_rack.ejectVelocity = 15

	-- Now add the MER model:
	populated_rack.Elements = {{
		ShapeName	=	rack_variant.shapename,
		IsAdapter   =   true,
	}}

	local connector_names = {
		"POINT_PYLON_06",
		"POINT_PYLON_01",
		"POINT_PYLON_04",
		"POINT_PYLON_02",
		"POINT_PYLON_05",
		"POINT_PYLON_03",
	}

	for i,v in ipairs(config) do
		populated_rack.Elements[#populated_rack.Elements + 1] = {
			connector_name = connector_names[v],
			payload_CLSID =	rocket_variant.CLSID
		}
	end

	return populated_rack
end

local function lau_117(element, clsid)

	swa_text = ""
	if string.find(clsid, "SWA") then
		swa_text = "(Special Weapons Adapter) "
	end

	local missile_variant = missile_data[element] or missile_data["AGM-65D"]
	local rack_variant = {mass = 59.0, shapename = "HB_F4E_LAU117", wstype = {4, 7, 32, WSTYPE_PLACEHOLDER} }
	local populated_rack = {}
	populated_rack.category           = CAT_MISSILES
	populated_rack.CLSID              = clsid
	populated_rack.attribute          = rack_variant.wstype
	populated_rack.Picture            = missile_variant.pic
	if missile_variant.PictureBlendColor then
		populated_rack.PictureBlendColor  = "0xffffffff"
	end
	populated_rack.Count              = 1
	populated_rack.displayName        = swa_text..missile_variant.name.." (LAU-117)"..swa_text
	populated_rack.wsTypeOfWeapon     = missile_variant.wstype
	populated_rack.Weight             = rack_variant.mass + missile_variant.mass
	populated_rack.Cx_pil             = missile_variant.cx + 0.00025

	populated_rack.Elements = {
		{
			ShapeName = rack_variant.shapename,
			IsAdapter = true,
		},
		{
			--Position = {-0.023782,	-0.486183,	0},
			--Position = {-0.035,	-0.340,	0},
			Position = {-0.072,	-0.198,	0},
			ShapeName = element
		}
	}

	return populated_rack
end

local function lau_88(element, count, clsid, side)
	local missile_variant = missile_data[element] or missile_data["AGM-65D"]
	local rack_variant = {mass = 211.0, shapename = "HB_ORD_LAU-88", wstype = {4, 7, 32, WSTYPE_PLACEHOLDER} } -- ACTUAL LAU WSTYPE is supposedly 4, 15, 47, but this stops DCS from starting...
	local populated_rack = {}

	populated_rack.category           = CAT_MISSILES
	populated_rack.CLSID              = clsid
	populated_rack.attribute          = rack_variant.wstype
	populated_rack.Picture            = missile_variant.pic
	if missile_variant.PictureBlendColor then
		populated_rack.PictureBlendColor  = "0xffffffff"
	end
	populated_rack.Count              = count
	populated_rack.displayName        = tostring(count).."x "..missile_variant.name.." (LAU-88)"
	populated_rack.wsTypeOfWeapon     = missile_variant.wstype
	populated_rack.Weight             = rack_variant.mass + count * missile_variant.mass
	populated_rack.Cx_pil             = missile_variant.cx * count + 0.001887

	-- Now add the LAU-88 model:
	populated_rack.Elements           =
	{
		{
			Position	=	{0,	0, 0}, -- Hmm, changing this seems to do nothing.
			ShapeName	=	rack_variant.shapename,
			IsAdapter   =   true,
		},
	}

	local positions = {
		{ missile_variant.offset, -0.113, -0.28 }, -- front left
		{ missile_variant.offset, -0.35, 0 }, -- front center
		{ missile_variant.offset, -.113, 0.28 } -- front right
	}

	local rotations =  {
		{ 90, 0, 0 }, -- left side
		{ 0,  0, 0 }, -- center
		{ -90, 0, 0 }, -- right side
	}

	-- currently order is the same for all sides; unsure if this is right.
	local order_left  = {1, 2, 3}
	local order_right = {3, 2, 1}
	local order_centre = {1, 3, 2}

	local order = order_centre

	if (side == "left") then
		order = order_left
	elseif (side == "right") then
		order = order_right
	end

	for i = 1,count do
		local missile_number = order[i]
		populated_rack.Elements[#populated_rack.Elements + 1] = {DrawArgs	=	{{1,1},{2,1}},
																 Position	=	positions[missile_number],
																 ShapeName	=	element,
																 Rotation	=   rotations[missile_number]}
	end

	return populated_rack
end

-- FROM BEHIND
-- 2-----3
--  \   /
--    1

-- Example Config
-- {1,1,1} -- Full
-- {1,0,0} -- Left Only
-- {0,1,0} -- Centre Only
function bru42_config(config)

	local result = {}

	local conversion = {2,1,3}

	for i,v in ipairs(config) do
		if v == 1 then
			result[#result+1] = conversion[i]
		end
	end

	 -- Sort so firing order is correct.
	 -- Last bomb on is fired first.
	table.sort(result, function(a,b) return a > b end )

	return result
end

declare_sensor({
	category = SENSOR_OPTICAL,
	Name = "AN/AVQ-23 Pave Spike - Sensor",
	type = OPTIC_SENSOR_IR,
	scan_volume =
	{
		azimuth = {-150.0, 150.0},
		elevation = {-150.0, 45.0}
	},
	view_volume_max = 
	{
		azimuth_sector = 18.4,
		elevation_sector = 24.1
	},
	magnifications = {3, 25, 90},
	resolution = 0.07, 
	laserRanger = true,
})

-- Drag calculated using ratio of Drag Index to known DCS drag value.
-- MK-83 Cx_pil = 0.00035
-- MK-83 index = 1.8
-- (Store index / MK-83 index) * MK-83 Cx_pil
-- Pave Spike Index = 1.3

function create_pave_spike(fast_track)

	local name = "HB_ORD_Pave_Spike"
	
	local fast_track_text = ""
	local fast_track_clsid = ""
	local fast_track_model = ""
	
	local picture = "Picto_PaveSpike.png"

	if fast_track then
		fast_track_text = " (Fast/Smart Track)"
		fast_track_clsid = "_FAST_TRACK"
		fast_track_model = "_Fast"
		picture = "Picto_PaveSpike_FAST_TRACK.png"
	end

	name = name..fast_track_model

	local result = {
		category = CAT_PODS,
		CLSID = "{HB_PAVE_SPIKE"..fast_track_clsid.."}",
		Picture = picture,
		displayName = _("AN/AVQ-23 Pave Spike"..fast_track_text.." - Targeting Pod"),
		Weight = 192, --kg
		Cx_pil = 0.000253,
		attribute = { wsType_Weapon, wsType_GContainer, wsType_Control_Cont, WSTYPE_PLACEHOLDER },
		shape_table_data = {
			{
				name = name,
				file = "",
				username = name,
				life = 1,
				fire = {0,1},
				index = WSTYPE_PLACEHOLDER,
			},
		},
		Elements = {
			[1] = {
				ShapeName = name,
			},
		},
		Sensors	 = 
		{
			OPTIC  = {"AN/AVQ-23 Pave Spike - Sensor"}
		}
	}

	return result
end

local pave_spike = create_pave_spike(false)
local pave_spike_fast = create_pave_spike(true)

declare_loadout(pave_spike)
declare_loadout(pave_spike_fast)

local alq131 = { --already exists in game so don't declare loadout
	CLSID	=	"{6D21ECEA-F85B-4E8D-9D51-31DC9B8AA4EF}",
	Picture	=	"ALQ131.png",
	displayName	=	_("ALQ-131 - ECM Pod"),
	Weight	=	305,
	Cx_pil	=	0.00083740234375,
	attribute	=	{4,	15,	45,	25},
	--PictureBlendColor = "0xffffffff",
	Elements	=	
	{
		[1]	=	
		{
			ShapeName	=	"ALQ-131",
		}, 
	}, -- end of Elements
}

function aero7_equipment_adapter(equipment, clsid)

	local adapter = {
		category = CAT_PODS,
		CLSID = clsid,
		attribute = {4, 7, 32, WSTYPE_PLACEHOLDER},
		Picture = equipment.Picture,
		--PictureBlendColor = "0xffffffff",
		Count = 1,
		displayName = equipment.displayName.." Rack",
		wsTypeOfWeapon = equipment.attribute,
		Weight = equipment.Weight + 25.9, --kg for adapter
		Cx_pil = equipment.Cx_pil,
		Elements = {
			{
				Position	=	{0,	0,	0},
				ShapeName	=	"HB_ORD_Missile_Well_Adapter",
				IsAdapter   =   true,
			},
			{
				ShapeName = equipment.Elements[1].ShapeName,
				payload_CLSID = equipment.CLSID,
				connector_name = "PNT_AIM7_4B",
			}
		},
	}
	return adapter
end

local bru42_all = bru42_config({1,1,1})

function bru_42(element, clsid, config)

	swa_text = ""
	if string.find(clsid, "SWA") then
		swa_text = "(Special Weapons Adapter) "
	end

	local config = config or {1,1,1}
	config = bru42_config(config)

	local count = #config
	local bomb_variant = bomb_data[element] or bomb_data["Mk-82"]
	local rack_variant = {mass = 128, shapename = "HB_F-4E_BRU-42", wstype = {4, 5, 32, WSTYPE_PLACEHOLDER}, cx = 0.00244 } -- Unsure what cx is reasonable for the rail
	local populated_rack = {}

	local connector_names =
	{
		"BRU-42_LOWER",
		"BRU-42_LEFT",
		"BRU-42_RIGHT"
	}

	populated_rack.category           = CAT_BOMBS
	populated_rack.CLSID              = clsid
	populated_rack.attribute          = rack_variant.wstype
	populated_rack.Picture            = bomb_variant.picture
	if bomb_variant.PictureBlendColor then
		populated_rack.PictureBlendColor  = "0xffffffff"
	end
	populated_rack.Count              = count
	populated_rack.displayName        = swa_text..tostring(count).."x "..bomb_variant.name.." (TER)"
	populated_rack.wsTypeOfWeapon     = bomb_variant.wsType
	populated_rack.Weight             = rack_variant.mass + count * bomb_variant.mass
	populated_rack.Cx_pil             = (bomb_variant.Cx * count) + rack_variant.cx
	populated_rack.settings = bomb_variant.settings

	-- Now add the MER model:
	populated_rack.Elements           =
	{
		{
			Position	=	{0,	0,	0},
			ShapeName	=	rack_variant.shapename,
			IsAdapter   =   true,
		},
	}

	for i,v in ipairs(config) do
		populated_rack.Elements[#populated_rack.Elements + 1] = {DrawArgs	=	{{1,1},{2,1}},
																 ShapeName	=	element,
																 --payload_CLSID	=	bomb_variant.CLSID,
																 connector_name = connector_names[v]}
	end

	return populated_rack
end


function bru_42_rocket(element, clsid, config)
	local config = config or {1,1,1}
	config = bru42_config(config)

	local count = #config
	local rocket_variant = rocket_data[element] or rocket_data["Mk-82"]
	local rack_variant = {mass = 128, shapename = "HB_F-4E_BRU-42", wstype = {4, 7, 32, WSTYPE_PLACEHOLDER}, cx = 0.00244 } -- Unsure what cx is reasonable for the rail
	local populated_rack = {}

	local connector_names =
	{
		"BRU-42_LOWER",
		"BRU-42_LEFT",
		"BRU-42_RIGHT"
	}

	populated_rack.category           = CAT_ROCKETS
	populated_rack.CLSID              = clsid
	populated_rack.attribute          = rack_variant.wstype
	populated_rack.Picture            = rocket_variant.picture
	if rocket_variant.PictureBlendColor then
		populated_rack.PictureBlendColor  = "0xffffffff"
	end
	populated_rack.Count              = count
	populated_rack.displayName        = tostring(count).."x "..rocket_variant.name.." (TER)"
	populated_rack.wsTypeOfWeapon     = rocket_variant.wsType
	populated_rack.Weight             = rack_variant.mass + count * rocket_variant.mass
	populated_rack.Weight_Empty 	  = rack_variant.mass + count * rocket_variant.mass_empty
	populated_rack.Cx_pil             = (rocket_variant.Cx * count) + rack_variant.cx

	-- Now add the MER model:
	populated_rack.Elements           =
	{
		{
			Position	=	{0,	0,	0},
			ShapeName	=	rack_variant.shapename,
			IsAdapter   =   true,
		},
	}

	for i,v in ipairs(config) do
		populated_rack.Elements[#populated_rack.Elements + 1] = {DrawArgs	=	{{1,1},{2,1}},
																 ShapeName	=	element,
																 payload_CLSID = rocket_variant.CLSID,
																 connector_name = connector_names[v]}
	end

	return populated_rack
end

-- Copied from F14!
local function belly_aim_7(clsid, missilename, wstype, mass)
	local ret = {
		category			=	CAT_AIR_TO_AIR,
		CLSID				=	clsid,
		Picture				=	"aim7m.png",
		Count			=	1,
		Weight			=	mass,
		attribute		=   wstype, --{4,	4,	7,	21},
		Cx_pil				=	(2.21 / 4096.0),--0.001959765625,
		-- AKO this value is almost 4x higher than 2.21/4096!
		-- note: cannot reduce belly AIM7 drag here without affecting missile flyout performance
		Count				=	1,
		Weight				=	230,
		Weight_Empty		=	0,
		Elements			=
		{
			{	ShapeName	=	"HB_F14_EXT_AIM-7"}
		}-- end of Elements
	}
	-- actually LAU-92 adapter, the LAU-92 is internal to the adapter and also internal in the F-14 belly sparrow slots
	--ret.displayName =	_("LAU-92").." "..missilename
	ret.displayName =	missilename

	declare_loadout(ret)
end

----- fuel tanks
local jp_4_standard_density = 6.5 * 0.453592 --kg/gal (ppg * kg/lb)

--declare_loadout(
--{
--    category		= CAT_FUEL_TANKS,
--    CLSID			= "{F14-300gal-empty}",
--    attribute		=  {wsType_Air,wsType_Free_Fall,wsType_FuelTank,WSTYPE_PLACEHOLDER},
--    Picture	=	"PTB.png",
--    Weight_Empty	= 50,
--    Weight			= 70, --20 eunusable or something
--    Capacity = 300*GALLON_TO_KG,
--    --attribute	=	{1,	3,	43,	12},
--    shape_table_data =
--    {
--        {
--            name	= "HB_F14_EXT_DROPTANK_EMPTY";
--            file	= "HB_F14_EXT_DROPTANK";
--            life	= 1;
--            fire	= { 0, 1};
--            username	= "Fuel tank 300 gal";
--            index	= WSTYPE_PLACEHOLDER;
--        },
--    },
--    Elements	=
--    {
--        [1]	=
--        {
--            Position	=	{0,	0,	0},
--            ShapeName	=	"HB_F14_EXT_DROPTANK_EMPTY",
--        },
--    }, -- end of Elements
--    displayName	=	_("Fuel tank 300 gal (empty)"),
--    Cx_pil = 0.0011,
--})


local bru42_opposed = {1,0,1}
local bru42_left = {1,1,0}
local bru42_right = {0,1,1}
local bru42_centre = {0,1,0}

local mer_left = {1,1,0, 1,1,0}
local mer_right = {0,1,1, 0,1,1}

local mer_forward = {1,1,1, 0,0,0}

local special_weapons_adapter_loadout = true

--[[
{HB_F4E_<bomb>_6x}
{HB_F4E_<bomb>_3x}
{HB_F4E_<bomb>_2x}
{HB_F4E_<bomb>_2x_SWA}
]]

function load_gp_bomb(bomb)
	declare_loadout(mer(bomb,"{HB_F4E_"..bomb.."_6x}"))
	declare_loadout(bru_42(bomb,"{HB_F4E_"..bomb.."_3x}"))
	declare_loadout(bru_42(bomb,"{HB_F4E_"..bomb.."_2x}",bru42_opposed))
	declare_loadout(bru_42(bomb,"{HB_F4E_"..bomb.."_2x_SWA}",bru42_opposed))
	declare_loadout(single_swa(bomb))
end

--[[
{HB_<bomb>_2x}
{HB_<bomb>_2x_SWA}
]]
function load_lg_bomb(bomb)
	declare_loadout(bru_42(bomb, "{HB_F4E_"..bomb.."_2x}", bru42_opposed))
	declare_loadout(bru_42(bomb, "{HB_F4E_"..bomb.."_2x_SWA}", bru42_opposed))
	declare_loadout(single_swa(bomb))
end


-- NOTE SWA in an CLSID is a reserved word just to make defining loadouts with the SWA easier

load_gp_bomb("MK-81")
load_gp_bomb("MK-82")
load_gp_bomb("MK-82_Snakeye")
load_gp_bomb("ROCKEYE")
load_gp_bomb("MK-82AIR")
load_gp_bomb("BDU-50LD")
load_gp_bomb("BDU-50HD")
load_gp_bomb("BDU-33")

load_lg_bomb("GBU-12")
load_lg_bomb("BDU_45LGB")


belly_aim_7("{HB_F4E_AIM-7F}", "AIM-7F", "weapons.missiles.AIM-7F", 231)
belly_aim_7("{HB_F4E_AIM-7M}", "AIM-7M", {wsType_Weapon,wsType_Missile,wsType_AA_Missile, AIM_7}, 231)
belly_aim_7("{HB_F4E_AIM-7E}", AIM_7E.displayName, AIM_7E.wsTypeOfWeapon, AIM_7E.mass)
belly_aim_7("{HB_F4E_AIM-7E-2}", AIM_7E_2.displayName, AIM_7E_2.wsTypeOfWeapon, AIM_7E_2.mass)


-- MK-83 Special Loadouts
--[[

MER
{HB_F4E_MK-83_MER_3x}
{HB_F4E_MK-83_MER_3x_Ripple}
{HB_F4E_MK-83_MER_2x}
{HB_F4E_MK-83_MER_1x_Left_Ripple}
{HB_F4E_MK-83_MER_1x_Right_Ripple}
TER
{HB_F4E_MK-83_3x}
{HB_F4E_MK-83_2x_Left}
{HB_F4E_MK-83_2x_Right}
]]
declare_loadout(mer("MK-83", "{HB_F4E_MK-83_MER_3x}", {0,1,0, 1,0,1}))
declare_loadout(mer("MK-83", "{HB_F4E_MK-83_MER_3x_Ripple}", {1,0,0, 0,1,1}, " Ripple"))
declare_loadout(mer("MK-83", "{HB_F4E_MK-83_MER_2x}", {1,0,1, 0,0,0}))
declare_loadout(mer("MK-83", "{HB_F4E_MK-83_MER_1x_Left_Ripple}", {1,0,0, 0,0,0}, " Ripple"))
declare_loadout(mer("MK-83", "{HB_F4E_MK-83_MER_1x_Right_Ripple}", {0,0,1, 0,0,0}, " Ripple"))

declare_loadout(bru_42("MK-83", "{HB_F4E_MK-83_3x}"))
declare_loadout(bru_42("MK-83", "{HB_F4E_MK-83_2x_Left}", bru42_left))
declare_loadout(bru_42("MK-83", "{HB_F4E_MK-83_2x_Right}", bru42_right))


--M117 Special Loadouts
--[[

MER
{HB_F4E_M117_MER_3x_Left}
{HB_F4E_M117_MER_3x_Right}
{HB_F4E_M117_MER_4x}
{HB_F4E_M117_MER_6x}
TER
{HB_F4E_M117_3x}
{HB_F4E_M117_2x_Left}
{HB_F4E_M117_2x_Right}
]]
declare_loadout(mer("M117", "{HB_F4E_M117_MER_3x_Left}", {1,0,0, 1,0,1}))
declare_loadout(mer("M117", "{HB_F4E_M117_MER_3x_Right}", {0,0,1, 1,0,1}))
declare_loadout(mer("M117", "{HB_F4E_M117_MER_5x}", {1,0,1, 1,1,1}))

declare_loadout(bru_42("M117", "{HB_F4E_M117_3x}"))
declare_loadout(bru_42("M117", "{HB_F4E_M117_2x_Left}",bru42_left))
declare_loadout(bru_42("M117", "{HB_F4E_M117_2x_Right}",bru42_right))

--CBU-87 Special Loadouts
--[[
MER
{HB_F4E_CBU-87_MER_4x}
{HB_F4E_CBU-87_MER_3x_Left}
{HB_F4E_CBU-87_MER_3x_Right}
TER
{HB_F4E_CBU-87_2x}
{HB_F4E_CBU-87_2x_SWA}
]]
declare_loadout(mer("CBU-87", "{HB_F4E_CBU-87_MER_4x}", {1,0,1, 1,0,1}))
declare_loadout(mer("CBU-87", "{HB_F4E_CBU-87_MER_3x_Left}", {1,0,0, 1,1,0}))
declare_loadout(mer("CBU-87", "{HB_F4E_CBU-87_MER_3x_Right}", {0,0,1, 0,1,1}))

declare_loadout(bru_42("CBU-87", "{HB_F4E_CBU-87_2x}",bru42_opposed))
declare_loadout(bru_42("CBU-87", "{HB_F4E_CBU-87_2x_SWA}",bru42_opposed))
declare_loadout(single_swa("CBU-87"))

--CBU-52B Special Loadouts
--[[
MER
{HB_F4E_CBU-52B_MER_6x}
{HB_F4E_CBU-52B_MER_3x_Left}
{HB_F4E_CBU-52B_MER_3x_Right}
TER
{HB_F4E_CBU-52B_2x}
{HB_F4E_CBU-52B_2x_SWA}
]]
declare_loadout(mer("CBU-52B", "{HB_F4E_CBU-52B_MER_6x}", {1,0,1, 1,0,1}))
declare_loadout(mer("CBU-52B", "{HB_F4E_CBU-52B_MER_3x_Left}", {0,1,1, 1,0,0}))
declare_loadout(mer("CBU-52B", "{HB_F4E_CBU-52B_MER_3x_Right}", {1,1,0, 0,0,1}))

declare_loadout(bru_42("CBU-52B", "{HB_F4E_CBU-52B_2x}",bru42_opposed))
declare_loadout(bru_42("CBU-52B", "{HB_F4E_CBU-52B_2x_SWA}",bru42_opposed))
declare_loadout(single_swa("CBU-52B"))

--BLU-107B Special Loadouts
--[[
MER
{HB_F4E_BLU-107B_MER_6x}
TER
{HB_F4E_BLU-107B_3x}
{HB_F4E_BLU-107B_3x_SWA}
]]
declare_loadout(mer("BLU-107B", "{HB_F4E_BLU-107B_6x}"))
declare_loadout(bru_42("BLU-107B", "{HB_F4E_BLU-107B_3x}"))
declare_loadout(bru_42("BLU-107B", "{HB_F4E_BLU-107B_3x_SWA}"))


--[[
{HB_F4E_<rockets>_2x_Right}
{HB_F4E_<rockets>_2x_Left}
{HB_F4E_<rockets>_2x_Centre}
{HB_F4E_<rockets>_3x}
{HB_F4E_<rockets>_MER_3x}
]]
function load_rockets(rockets)
	declare_loadout(bru_42_rocket(rockets, "{HB_F4E_"..rockets.."_2x_Right}", bru42_right))
	declare_loadout(bru_42_rocket(rockets, "{HB_F4E_"..rockets.."_2x_Left}", bru42_left))
	declare_loadout(bru_42_rocket(rockets, "{HB_F4E_"..rockets.."_1x}", bru42_centre))
	declare_loadout(bru_42_rocket(rockets, "{HB_F4E_"..rockets.."_3x}"))
	declare_loadout(mer_rocket(rockets, "{HB_F4E_"..rockets.."_MER_3x}", mer_forward))
end

load_rockets("LAU-68_MK5")
load_rockets("LAU-68_MK1")
load_rockets("LAU-68_WP156")

load_rockets("LAU-3_MK5")
load_rockets("LAU-3_MK1")
load_rockets("LAU-3_WP156")

declare_loadout(mer_dispenser("SUU-25", "{HB_F4E_SUU-25_MER_2x_Left}", {0,0,0,1,1,0}))
declare_loadout(mer_dispenser("SUU-25", "{HB_F4E_SUU-25_MER_2x_Right}", {0,0,0,0,1,1}))
declare_loadout(mer_dispenser("SUU-25", "{HB_F4E_SUU-25_MER_2x}", {0,0,0,1,0,1}))

--[[
{HB_F4E_<cbu_dispenser>_MER_2x_Right}
{HB_F4E_<cbu_dispenser>_MER_2x_Left}
{HB_F4E_<cbu_dispenser>_MER_1x_Right}
{HB_F4E_<cbu_dispenser>_MER_1x_Left}
{HB_F4E_<cbu_dispenser>_MER_1x}
]]
function load_cbu(cbu_dispenser)
	declare_loadout(mer_dispenser(cbu_dispenser, "{HB_F4E_"..cbu_dispenser.."_MER_2x_Left}", 	{0,0,0,1,1,0}))
	declare_loadout(mer_dispenser(cbu_dispenser, "{HB_F4E_"..cbu_dispenser.."_MER_2x_Right}", 	{0,0,0,0,1,1}))
	declare_loadout(mer_dispenser(cbu_dispenser, "{HB_F4E_"..cbu_dispenser.."_MER_1x_Left}", 	{0,0,0,1,0,0}))
	declare_loadout(mer_dispenser(cbu_dispenser, "{HB_F4E_"..cbu_dispenser.."_MER_1x_Right}", 	{0,0,0,0,0,1}))
	declare_loadout(mer_dispenser(cbu_dispenser, "{HB_F4E_"..cbu_dispenser.."_MER_1x}", 		{0,0,0,0,1,0}))
end

load_cbu("CBU-1A")
load_cbu("CBU-2A")
load_cbu("CBU-2BA")


--[[
{HB_F4E_<maverick>_LAU88_3x_Right}
{HB_F4E_<maverick>_LAU88_3x_Left}
{HB_F4E_<maverick>_LAU88_2x_Right}
{HB_F4E_<maverick>_LAU88_2x_Left}
{HB_F4E_<maverick>_LAU117_SWA}
{HB_F4E_<maverick>_LAU117_SWA}
]]
function load_maverick(maverick)

	-- No idea why there are two loadouts for 3x mavericks but not changing it now.

	declare_loadout(lau_88(maverick, 3, "{HB_F4E"..maverick.."_LAU88_3x_Right}", "right"))
	declare_loadout(lau_88(maverick, 3, "{HB_F4E"..maverick.."_LAU88_3x_Left}", "left"))

	declare_loadout(lau_88(maverick, 2, "{HB_F4E"..maverick.."_LAU88_2x_Right}", "right"))
	declare_loadout(lau_88(maverick, 2, "{HB_F4E"..maverick.."_LAU88_2x_Left}", "left"))

	declare_loadout(lau_117(maverick, "{HB_F4E_"..maverick.."_LAU117}"))
	declare_loadout(lau_117(maverick, "{HB_F4E_"..maverick.."_LAU117_SWA}"))
end

load_maverick("AGM-65A")
load_maverick("AGM-65B")
load_maverick("AGM-65D")

declare_loadout(lau_117("AGM-65G", "{HB_F4E_AGM-65G_LAU117}"))

declare_loadout(aero7_equipment_adapter(pave_spike, "{HB_PAVE_SPIKE_ON_ADAPTER_IN_AERO7}"))
declare_loadout(aero7_equipment_adapter(pave_spike_fast, "{HB_PAVE_SPIKE_FAST_ON_ADAPTER_IN_AERO7}"))
declare_loadout(aero7_equipment_adapter(alq131, "{HB_ALQ-131_ON_ADAPTER_IN_AERO7}"))
